package tech.tablesaw.mapping;

import java.time.LocalTime;
import java.time.temporal.ChronoUnit;

import tech.tablesaw.api.IntColumn;
import tech.tablesaw.api.LongColumn;
import tech.tablesaw.api.ShortColumn;
import tech.tablesaw.api.TimeColumn;
import tech.tablesaw.columns.TimeColumnUtils;
import tech.tablesaw.columns.packeddata.PackedLocalTime;

public interface TimeMapUtils extends TimeColumnUtils {

    default LongColumn differenceInMilliseconds(TimeColumn column2) {
        return difference(column2, ChronoUnit.MILLIS);
    }

    default LongColumn differenceInSeconds(TimeColumn column2) {
        return difference(column2, ChronoUnit.SECONDS);
    }

    default LongColumn differenceInMinutes(TimeColumn column2) {
        return difference(column2, ChronoUnit.MINUTES);
    }

    default LongColumn differenceInHours(TimeColumn column2) {
        return difference(column2, ChronoUnit.HOURS);
    }

    default LongColumn difference(TimeColumn column2, ChronoUnit unit) {
        LongColumn newColumn = new LongColumn(name() + " - " + column2.name());

        for (int r = 0; r < size(); r++) {
            int c1 = this.getInt(r);
            int c2 = column2.getInt(r);
            if (c1 == TimeColumn.MISSING_VALUE || c2 == TimeColumn.MISSING_VALUE) {
                newColumn.append(IntColumn.MISSING_VALUE);
            } else {
                newColumn.append(difference(c1, c2, unit));
            }
        }
        return newColumn;
    }

    default long difference(int packedLocalTime1, int packedLocalTime2, ChronoUnit unit) {
        LocalTime value1 = PackedLocalTime.asLocalTime(packedLocalTime1);
        LocalTime value2 = PackedLocalTime.asLocalTime(packedLocalTime2);
        return unit.between(value1, value2);
    }

    default ShortColumn hour() {
        ShortColumn newColumn = new ShortColumn(name() + "[" + "hour" + "]");
        for (int r = 0; r < size(); r++) {
            int c1 = getInt(r);
            if (c1 != TimeColumn.MISSING_VALUE) {
                newColumn.append(PackedLocalTime.getHour(c1));
            } else {
                newColumn.append(ShortColumn.MISSING_VALUE);
            }
        }
        return newColumn;
    }

    default IntColumn minuteOfDay() {
        IntColumn newColumn = new IntColumn(name() + "[" + "minute-of-day" + "]");
        for (int r = 0; r < size(); r++) {
            int c1 = getInt(r);
            if (c1 != TimeColumn.MISSING_VALUE) {
                newColumn.append(PackedLocalTime.getMinuteOfDay(c1));
            } else {
                newColumn.append(IntColumn.MISSING_VALUE);
            }
        }
        return newColumn;
    }

    default IntColumn secondOfDay() {
        IntColumn newColumn = new IntColumn(name() + "[" + "second-of-day" + "]");
        for (int r = 0; r < size(); r++) {
            int c1 = getInt(r);
            if (c1 != TimeColumn.MISSING_VALUE) {
                newColumn.append(PackedLocalTime.getSecondOfDay(c1));
            } else {
                newColumn.append(IntColumn.MISSING_VALUE);
            }
        }
        return newColumn;
    }

    LocalTime get(int r);

    int getInt(int r);
}
