/*
 * Decompiled with CFR 0.152.
 */
package tech.tablesaw.io.xlsx;

import com.google.common.collect.Iterables;
import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.concurrent.Immutable;
import org.apache.poi.ss.format.CellDateFormatter;
import org.apache.poi.ss.format.CellGeneralFormatter;
import org.apache.poi.ss.format.CellNumberFormatter;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import tech.tablesaw.api.ColumnType;
import tech.tablesaw.api.DoubleColumn;
import tech.tablesaw.api.LongColumn;
import tech.tablesaw.api.Table;
import tech.tablesaw.columns.Column;
import tech.tablesaw.io.DataReader;
import tech.tablesaw.io.ReaderRegistry;
import tech.tablesaw.io.RuntimeIOException;
import tech.tablesaw.io.Source;
import tech.tablesaw.io.xlsx.XlsxReadOptions;

@Immutable
public class XlsxReader
implements DataReader<XlsxReadOptions> {
    private static final XlsxReader INSTANCE = new XlsxReader();

    public static void register(ReaderRegistry registry) {
        registry.registerExtension("xlsx", (DataReader)INSTANCE);
        registry.registerMimeType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", (DataReader)INSTANCE);
        registry.registerOptions(XlsxReadOptions.class, (DataReader)INSTANCE);
    }

    public Table read(XlsxReadOptions options) {
        List<Table> tables = null;
        try {
            tables = this.readMultiple(options, true);
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
        if (options.sheetIndex() != null) {
            int index = options.sheetIndex();
            if (index < 0 || index >= tables.size()) {
                throw new IndexOutOfBoundsException(String.format("Sheet index %d outside bounds. %d sheets found.", index, tables.size()));
            }
            Table table = tables.get(index);
            if (table == null) {
                throw new IllegalArgumentException(String.format("No table found at sheet index %d.", index));
            }
            return table;
        }
        return tables.stream().filter(t -> t != null).findFirst().orElseThrow(() -> new IllegalArgumentException("No tables found."));
    }

    public List<Table> readMultiple(XlsxReadOptions options) throws IOException {
        return this.readMultiple(options, false);
    }

    /*
     * Loose catch block
     */
    protected List<Table> readMultiple(XlsxReadOptions options, boolean includeNulls) throws IOException {
        byte[] bytes = null;
        InputStream input = this.getInputStream(options, bytes);
        ArrayList<Table> tables = new ArrayList<Table>();
        try {
            try (XSSFWorkbook workbook = new XSSFWorkbook(input);){
                for (Sheet sheet : workbook) {
                    TableRange tableArea = this.findTableArea(sheet);
                    if (tableArea != null) {
                        Table table = this.createTable(sheet, tableArea, options);
                        tables.add(table);
                        continue;
                    }
                    if (!includeNulls) continue;
                    tables.add(null);
                }
                ArrayList<Table> arrayList = tables;
                return arrayList;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            if (options.source().reader() == null) {
                input.close();
            }
        }
    }

    private Boolean isBlank(Cell cell) {
        switch (cell.getCellType()) {
            case STRING: {
                if (cell.getRichStringCellValue().length() <= 0) break;
                return false;
            }
            case NUMERIC: {
                if (!(DateUtil.isCellDateFormatted((Cell)cell) ? cell.getDateCellValue() != null : cell.getNumericCellValue() != 0.0)) break;
                return false;
            }
            case BOOLEAN: {
                if (!cell.getBooleanCellValue()) break;
                return false;
            }
            case BLANK: {
                return true;
            }
        }
        return null;
    }

    private TableRange findTableArea(Sheet sheet) {
        int row1 = -1;
        int row2 = -1;
        TableRange lastRowArea = null;
        for (Row row : sheet) {
            TableRange rowArea = this.findRowArea(row);
            if (lastRowArea == null && rowArea != null) {
                if (row1 >= 0) continue;
                lastRowArea = rowArea;
                row2 = row1 = row.getRowNum();
                continue;
            }
            if (lastRowArea != null && rowArea == null) {
                if (row2 > row1) break;
                row1 = -1;
                continue;
            }
            if (lastRowArea == null && rowArea == null) {
                row1 = -1;
                continue;
            }
            if (rowArea.startColumn < lastRowArea.startColumn || rowArea.endColumn > lastRowArea.endColumn) {
                lastRowArea = null;
                row2 = -1;
                continue;
            }
            row2 = row.getRowNum();
        }
        return row1 >= 0 && lastRowArea != null ? new TableRange(row1, row2, lastRowArea.startColumn, lastRowArea.endColumn) : null;
    }

    private TableRange findRowArea(Row row) {
        int col1 = -1;
        int col2 = -1;
        for (Cell cell : row) {
            Boolean blank = this.isBlank(cell);
            if (col1 < 0 && Boolean.FALSE.equals(blank)) {
                col2 = col1 = cell.getColumnIndex();
                continue;
            }
            if (col1 < 0 || col2 < col1) continue;
            if (Boolean.FALSE.equals(blank)) {
                col2 = cell.getColumnIndex();
                continue;
            }
            if (!Boolean.TRUE.equals(blank)) continue;
            break;
        }
        return col1 >= 0 && col2 >= col1 ? new TableRange(0, 0, col1, col2) : null;
    }

    private InputStream getInputStream(XlsxReadOptions options, byte[] bytes) throws FileNotFoundException {
        if (bytes != null) {
            return new ByteArrayInputStream(bytes);
        }
        if (options.source().inputStream() != null) {
            return options.source().inputStream();
        }
        return new FileInputStream(options.source().file());
    }

    private Table createTable(Sheet sheet, TableRange tableArea, XlsxReadOptions options) {
        Optional<List<String>> optHeaderNames = this.getHeaderNames(sheet, tableArea);
        optHeaderNames.ifPresent(h -> tableArea.startRow++);
        List<String> headerNames = optHeaderNames.orElse(this.calculateDefaultColumnNames(tableArea));
        Table table = Table.create((String)(options.tableName() + "#" + sheet.getSheetName()));
        ArrayList<Object> columns = new ArrayList<Object>(Collections.nCopies(headerNames.size(), null));
        for (int rowNum = tableArea.startRow; rowNum <= tableArea.endRow; ++rowNum) {
            Row row = sheet.getRow(rowNum);
            for (int colNum = 0; colNum < headerNames.size(); ++colNum) {
                int excelColNum = colNum + tableArea.startColumn;
                Cell cell = row.getCell(excelColNum, Row.MissingCellPolicy.RETURN_BLANK_AS_NULL);
                Column column = (Column)columns.get(colNum);
                String columnName = headerNames.get(colNum);
                if (cell != null) {
                    Column<?> altColumn;
                    if (column == null) {
                        column = this.createColumn(colNum, columnName, sheet, excelColNum, tableArea, options);
                        columns.set(colNum, column);
                        while (column.size() < rowNum - tableArea.startRow) {
                            column.appendMissing();
                        }
                    }
                    if ((altColumn = this.appendValue(column, cell)) != null && altColumn != column) {
                        column = altColumn;
                        columns.set(colNum, column);
                    }
                } else {
                    boolean hasCustomizedType = options.columnTypeReadOptions().columnType(colNum, columnName).isPresent();
                    if (column == null && hasCustomizedType) {
                        ColumnType columnType = (ColumnType)options.columnTypeReadOptions().columnType(colNum, columnName).get();
                        column = columnType.create(columnName).appendMissing();
                        columns.set(colNum, column);
                    } else if (hasCustomizedType) {
                        column.appendMissing();
                    }
                }
                if (column == null) continue;
                while (column.size() <= rowNum - tableArea.startRow) {
                    column.appendMissing();
                }
            }
        }
        columns.removeAll(Collections.singleton(null));
        table.addColumns(columns.toArray(new Column[columns.size()]));
        return table;
    }

    private Optional<List<String>> getHeaderNames(Sheet sheet, TableRange tableArea) {
        Row row = sheet.getRow(tableArea.startRow);
        List headerNames = IntStream.range(tableArea.startColumn, tableArea.endColumn + 1).mapToObj(arg_0 -> ((Row)row).getCell(arg_0)).filter(cell -> cell.getCellType() == CellType.STRING).map(cell -> cell.getRichStringCellValue().getString()).collect(Collectors.toList());
        return headerNames.size() == tableArea.getColumnCount() ? Optional.of(headerNames) : Optional.empty();
    }

    private List<String> calculateDefaultColumnNames(TableRange tableArea) {
        return IntStream.range(tableArea.startColumn, tableArea.endColumn + 1).mapToObj(i -> "col" + i).collect(Collectors.toList());
    }

    private Column<?> appendValue(Column<?> column, Cell cell) {
        CellType cellType = cell.getCellType() == CellType.FORMULA ? cell.getCachedFormulaResultType() : cell.getCellType();
        switch (cellType) {
            case STRING: {
                column.appendCell(cell.getRichStringCellValue().getString());
                return null;
            }
            case NUMERIC: {
                if (DateUtil.isCellDateFormatted((Cell)cell)) {
                    Date date = cell.getDateCellValue();
                    LocalDateTime localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
                    if (column.type() == ColumnType.STRING) {
                        String dataFormatStyle = cell.getCellStyle().getDataFormatString();
                        String val = "general".equalsIgnoreCase(dataFormatStyle) ? new CellGeneralFormatter().format((Object)cell.getNumericCellValue()) : new CellDateFormatter(dataFormatStyle).format((Object)cell.getDateCellValue());
                        column.appendCell(val);
                    } else {
                        column.appendCell(localDate.toString());
                    }
                    return null;
                }
                double num = cell.getNumericCellValue();
                if (column.type() == ColumnType.INTEGER) {
                    Column<?> intColumn = column;
                    if ((double)((int)num) == num) {
                        intColumn.append((Object)((int)num));
                        return null;
                    }
                    if ((double)((long)num) == num) {
                        LongColumn altColumn = LongColumn.create((String)column.name(), (int)column.size());
                        altColumn = intColumn.mapInto(s -> (long)s, (Column)altColumn);
                        altColumn.append((Object)((long)num));
                        return altColumn;
                    }
                    DoubleColumn altColumn = DoubleColumn.create((String)column.name(), (int)column.size());
                    altColumn = intColumn.mapInto(s -> (double)s, (Column)altColumn);
                    altColumn.append((Object)num);
                    return altColumn;
                }
                if (column.type() == ColumnType.LONG) {
                    Column<?> longColumn = column;
                    if ((double)((long)num) == num) {
                        longColumn.append((Object)((long)num));
                        return null;
                    }
                    DoubleColumn altColumn = DoubleColumn.create((String)column.name(), (int)column.size());
                    altColumn = longColumn.mapInto(s -> (double)s, (Column)altColumn);
                    altColumn.append((Object)num);
                    return altColumn;
                }
                if (column.type() == ColumnType.DOUBLE) {
                    Column<?> doubleColumn = column;
                    doubleColumn.append((Object)num);
                    return null;
                }
                if (column.type() != ColumnType.STRING) break;
                Column<?> stringColumn = column;
                String dataFormatStyle = cell.getCellStyle().getDataFormatString();
                String val = "general".equalsIgnoreCase(dataFormatStyle) ? new CellGeneralFormatter().format((Object)cell.getNumericCellValue()) : new CellNumberFormatter(dataFormatStyle).format((Object)cell.getNumericCellValue());
                stringColumn.append((Object)val);
                break;
            }
            case BOOLEAN: {
                if (column.type() == ColumnType.BOOLEAN) {
                    Column<?> booleanColumn = column;
                    booleanColumn.append((Object)cell.getBooleanCellValue());
                    return null;
                }
                if (column.type() != ColumnType.STRING) break;
                Column<?> stringColumn = column;
                String val = new CellGeneralFormatter().format((Object)cell.getBooleanCellValue());
                stringColumn.append((Object)val);
            }
        }
        return null;
    }

    private Column<?> createColumn(int colNum, String name, Sheet sheet, int excelColNum, TableRange tableRange, XlsxReadOptions options) {
        ColumnType columnType = options.columnTypeReadOptions().columnType(colNum, name).orElse(this.calculateColumnTypeForColumn(sheet, excelColNum, tableRange).orElse((ColumnType)ColumnType.STRING));
        Column column = columnType.create(name);
        return column;
    }

    public Table read(Source source) {
        return this.read(XlsxReadOptions.builder(source).build());
    }

    private Optional<ColumnType> calculateColumnTypeForColumn(Sheet sheet, int col, TableRange tableRange) {
        Set<CellType> cellTypes = this.getCellTypes(sheet, col, tableRange);
        if (cellTypes.size() != 1) {
            return Optional.empty();
        }
        CellType cellType = (CellType)Iterables.get(cellTypes, (int)0);
        switch (cellType) {
            case STRING: {
                return Optional.of(ColumnType.STRING);
            }
            case NUMERIC: {
                return this.allNumericFieldsDateFormatted(sheet, col, tableRange) ? Optional.of(ColumnType.LOCAL_DATE_TIME) : Optional.of(ColumnType.INTEGER);
            }
            case BOOLEAN: {
                return Optional.of(ColumnType.BOOLEAN);
            }
        }
        return Optional.empty();
    }

    private Set<CellType> getCellTypes(Sheet sheet, int col, TableRange tableRange) {
        return IntStream.range(tableRange.startRow, tableRange.endRow + 1).mapToObj(arg_0 -> ((Sheet)sheet).getRow(arg_0)).filter(Objects::nonNull).map(row -> row.getCell(col)).filter(Objects::nonNull).filter(cell -> Optional.ofNullable(this.isBlank((Cell)cell)).orElse(false) == false).map(cell -> cell.getCellType() == CellType.FORMULA ? cell.getCachedFormulaResultType() : cell.getCellType()).collect(Collectors.toSet());
    }

    private boolean allNumericFieldsDateFormatted(Sheet sheet, int col, TableRange tableRange) {
        return IntStream.range(tableRange.startRow, tableRange.endRow + 1).mapToObj(arg_0 -> ((Sheet)sheet).getRow(arg_0)).filter(Objects::nonNull).map(row -> row.getCell(col)).filter(Objects::nonNull).filter(cell -> cell.getCellType() == CellType.NUMERIC || cell.getCellType() == CellType.FORMULA && cell.getCachedFormulaResultType() == CellType.NUMERIC).allMatch(DateUtil::isCellDateFormatted);
    }

    static {
        XlsxReader.register(Table.defaultReaderRegistry);
    }

    private static class TableRange {
        private int startRow;
        private int endRow;
        private int startColumn;
        private int endColumn;

        TableRange(int startRow, int endRow, int startColumn, int endColumn) {
            this.startRow = startRow;
            this.endRow = endRow;
            this.startColumn = startColumn;
            this.endColumn = endColumn;
        }

        public int getColumnCount() {
            return this.endColumn - this.startColumn + 1;
        }
    }
}

