/*
 * Units of Measurement TCK
 * Copyright © 2005-2023, Jean-Marie Dautelle, Werner Keil, Otavio Santana.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of JSR-385 nor the names of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package tech.units.tck.tests.spi;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNot.not;
import static org.testng.Assert.assertNotNull;
import static tech.units.tck.TCKRunner.SECTION_PREFIX;
import static tech.units.tck.TCKRunner.SPEC_ID;
import static tech.units.tck.TCKRunner.SPEC_VERSION;
import static tech.units.tck.util.TestGroups.SPI;

import java.util.Set;

import javax.measure.Dimension;
import javax.measure.Unit;
import javax.measure.spi.ServiceProvider;
import javax.measure.spi.SystemOfUnits;

import org.jboss.test.audit.annotations.SpecAssertion;
import org.jboss.test.audit.annotations.SpecVersion;
import org.testng.annotations.Test;

import tech.units.tck.util.TestUtils;

/**
 * Tests for SystemOfUnits
 *
 * @author <a href="mailto:werner@units.tech">Werner Keil</a>
 * @version 2.3, October 4, 2023
 * @since 1.0
 */
@SpecVersion(spec = SPEC_ID, version = SPEC_VERSION)
public class SystemOfUnitsTest {
    private static final String SECTION_NUM = "5.2";

    /**
     * Ensure at least one SystemOfUnits implementation exists.
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A1")
    @Test(groups = { SPI }, description = SECTION_NUM
	    + " Ensure a SystemOfUnits implementation exists for every ServiceProvider")
    public void testEnsureGotSystemOfUnits() {
    	for (ServiceProvider provider : ServiceProvider.available()) {
    	    assertNotNull(provider,	SECTION_PREFIX + SECTION_NUM + ": ServiceProvider is null");
    	    assertNotNull(provider.getSystemOfUnitsService(), String.format("SystemOfUnitsService is null for %s", provider));
    	    assertNotNull(provider.getSystemOfUnitsService().getAvailableSystemsOfUnits(), 
    	    		String.format("Available SystemOfUnits is null for %s", provider));
    	    assertThat(String.format("SystemOfUnits not found in %s", provider), 
    	    		provider.getSystemOfUnitsService().getAvailableSystemsOfUnits(), is(not(empty())));
    	}
    }

    /**
     * Ensure the getName() method is implemented.
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A2")
    @Test(groups = { SPI }, description = SECTION_NUM
	    + " Ensure the getName() method is implemented.")
    public void testSystemOfUnitsGetName() {
    	for (SystemOfUnits suo : ServiceProvider.current()
    		.getSystemOfUnitsService().getAvailableSystemsOfUnits()) {
    	    Class<?> type = suo.getClass();
    	    TestUtils
    		    .testHasPublicMethod(SECTION_PREFIX + SECTION_NUM, type, "getName");
    	}
    }

    /**
     * Ensure the getUnit() method is implemented.
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A3")
    @Test(groups = { SPI }, description = SECTION_NUM
	    + " Ensure the getUnit() method is implemented.")
    public void testSystemOfUnitsGetUnit() {
    	for (SystemOfUnits suo : ServiceProvider.current()
    		.getSystemOfUnitsService().getAvailableSystemsOfUnits()) {
    	    Class<?> type = suo.getClass();
    	    TestUtils.testHasPublicMethod(SECTION_PREFIX + SECTION_NUM, type, true,
    		    Unit.class, "getUnit", Class.class);
    	}
    }
    
    /**
     * Ensure the getUnit() method is implemented.
     * @since 2.0
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A4")
    @Test(groups = { SPI }, description = SECTION_NUM
        + " Ensure the getUnit() method is implemented.")
    public void testSystemOfUnitsGetUnitForString() {
        for (SystemOfUnits suo : ServiceProvider.current()
            .getSystemOfUnitsService().getAvailableSystemsOfUnits()) {
            Class<?> type = suo.getClass();
            TestUtils.testHasPublicMethod(SECTION_PREFIX + SECTION_NUM, type, true,
                Unit.class, "getUnit", String.class);
        }
    }

    /**
     * Ensure the getUnits() method is implemented.
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A5")
    @Test(groups = { SPI }, description = SECTION_NUM
	    + " Ensure the getUnits() method is implemented.")
    public void testSystemOfUnitsGetUnits() {
    	for (SystemOfUnits suo : ServiceProvider.current()
    		.getSystemOfUnitsService().getAvailableSystemsOfUnits()) {
    	    Class<?> type = suo.getClass();
    	    TestUtils.testHasPublicMethod(SECTION_PREFIX + SECTION_NUM, type,
    		    "getUnits", false);
    	}
    }

    /**
     * Ensure the getUnits() method is implemented.
     */
    @SpecAssertion(section = SECTION_NUM, id = "52-A6")
    @Test(groups = { SPI }, description = "5.2 Ensure the getUnits() method is implemented.")
    public void testSystemOfUnitsGetUnitsForDimension() {
    	for (SystemOfUnits suo : ServiceProvider.current()
    		.getSystemOfUnitsService().getAvailableSystemsOfUnits()) {
    	    Class<?> type = suo.getClass();
    	    TestUtils.testHasPublicMethod(SECTION_PREFIX + SECTION_NUM, type, true, Set.class,
    		    "getUnits", Dimension.class);
    	}
    }
}
