package top.lshaci.framework.common.model;

import lombok.Data;
import lombok.Getter;
import lombok.experimental.Accessors;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>Web controller json response</p><br><br>
 *
 * <p>0.0.4:</p>
 * <pre>
 *     Add method: successMessage and message; Change the name of the params field to otherData<br>
 * </pre>
 * <p>1.0.5:</p>
 * <pre>
 *     Move form framework web module to framework common module
 * </pre>
 * <p>1.0.9:</p>
 * <pre>
 *     1.使用继承hashmap的方式
 *     2.修改方法名
 * </pre>
 * <p>1.1.0:</p>
 * <pre>
 *     1.修改方法名为 1.0.5 版本
 *     2.Field: otherData 字段添加 final 修饰
 *     3.Method: otherData 方法入参修改为 public JsonResponse&lt;R&gt; otherData(Map&lt;String, ? extends Object&gt; otherData)
 * </pre>
 *
 * @author lshaci
 * @version 1.1.0
 * @since 0.0.1
 */
@Data
@Accessors(chain = true)
public class JsonResponse<R> implements Serializable {

    private static final long serialVersionUID = 7574078101944305355L;

    /**
     * 默认失败响应码
     */
    private static final int default_failure_code = 500_0_00;

    /**
     * 默认失败响应消息
     */
    private static final String default_failure_message = "系统繁忙, 请稍后重试";
    /**
     * 默认成功响应消息
     */
    private static final String default_success_message = "success";

    /**
     * 状态信息; true: 成功, false: 失败
     */
    private boolean status;

    /**
     * 响应码
     */
    private int code;

    /**
     * 响应消息
     */
    private String message;

    /**
     * 需要返回的数据
     */
    private R data;

    /**
     * 返回添加的额外数据
     */
    @Getter
    private final Map<String, Object> otherData = new HashMap<>();

    /**
     * Build a json response
     *
     * @param <R> 响应数据类型
     * @return json response
     */
    public static <R> JsonResponse<R> build() {
        return new JsonResponse<>();
    }

    /**
     * 失败响应
     * <pre>
     * {
     *     "code":500,
     *     "status:false,
     *     "message":"未知异常, 请联系管理员"
     * }
     * </pre>
     *
     * @param <R> 响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> failure() {
        return failure(default_failure_message);
    }

    /**
     * 失败响应
     * <pre>
     * {
     *     "code":500,
     *     "status:false,
     * }
     * </pre>
     *
     * @param message 失败消息
     * @param <R>     响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> failure(String message) {
        return failure(default_failure_code, message);
    }

    /**
     * 失败响应
     *
     * @param code    失败响应码
     * @param message 失败消息
     * @param <R>     响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> failure(int code, String message) {
        final JsonResponse<R> jsonResponse = build();
        jsonResponse.status = false;
        jsonResponse.code = code;
        jsonResponse.message = message;
        return jsonResponse;
    }

    /**
     * 成功响应
     * <pre>
     * {
     *     "code":0,
     *     "status:true,
     *     "message":"success"
     * }
     * </pre>
     *
     * @param <R> 响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> success() {
        final JsonResponse<R> jsonResponse = build();
        jsonResponse.status = true;
        jsonResponse.message = default_success_message;
        return jsonResponse;
    }

    /**
     * 成功响应数据
     *
     * @param data 数据
     * @param <R>  响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> success(R data) {
        final JsonResponse<R> jsonResponse = success();
        jsonResponse.data = data;
        return jsonResponse;
    }

    /**
     * 成功响应消息
     *
     * @param message 消息
     * @param <R>     响应数据类型
     * @return 响应
     */
    public static <R> JsonResponse<R> successMessage(String message) {
        final JsonResponse<R> jsonResponse = success();
        jsonResponse.message = message;
        return jsonResponse;
    }

    /**
     * 响应数据
     *
     * @param data 数据
     * @return 响应
     */
    public JsonResponse<R> data(R data) {
        this.data = data;
        return this;
    }

    /**
     * 响应其它数据
     *
     * @param otherData 其它数据
     * @return 响应
     */
    public JsonResponse<R> otherData(Map<String, ? extends Object> otherData) {
        this.otherData.putAll(otherData);
        return this;
    }

    /**
     * 响应数据
     *
     * @param key   数据键
     * @param value 数据值
     * @return 响应
     */
    public JsonResponse<R> otherData(String key, Object value) {
        this.otherData.put(key, value);
        return this;
    }

    /**
     * 响应消息
     *
     * @param message 消息
     * @return 响应
     */
    public JsonResponse<R> message(String message) {
        this.message = message;
        return this;
    }

    /**
     * 响应码
     *
     * @param code 响应码
     * @return 响应
     */
    public JsonResponse<R> code(int code) {
        this.code = code;
        return this;
    }

}
