/*
 * Decompiled with CFR 0.152.
 */
package uk.co.spudsoft.jwtvalidatorvertx.impl;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import io.vertx.core.Future;
import io.vertx.ext.auth.impl.jose.JWK;
import io.vertx.ext.auth.impl.jose.JWS;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Base64;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.co.spudsoft.jwtvalidatorvertx.IssuerAcceptabilityHandler;
import uk.co.spudsoft.jwtvalidatorvertx.JsonWebKeySetHandler;
import uk.co.spudsoft.jwtvalidatorvertx.Jwt;
import uk.co.spudsoft.jwtvalidatorvertx.JwtValidator;

public class JwtValidatorVertxImpl
implements JwtValidator {
    private static final Logger logger = LoggerFactory.getLogger(JwtValidatorVertxImpl.class);
    private static final Base64.Decoder B64DECODER = Base64.getUrlDecoder();
    private static final Set<String> DEFAULT_PERMITTED_ALGS = ImmutableSet.of((Object)"EdDSA", (Object)"ES256", (Object)"ES384", (Object)"ES512", (Object)"PS256", (Object)"PS384", (Object[])new String[]{"PS512", "ES256K", "RS256", "RS384", "RS512"});
    private Set<String> permittedAlgs;
    private boolean requireExp = true;
    private boolean requireNbf = true;
    private long timeLeewayMilliseconds = 0L;
    private final JsonWebKeySetHandler jsonWebKeySetHandler;
    private final IssuerAcceptabilityHandler issuerAcceptabilityHandler;

    public JwtValidatorVertxImpl(JsonWebKeySetHandler jsonWebKeySetHandler, IssuerAcceptabilityHandler issuerAcceptabilityHandler) {
        this.jsonWebKeySetHandler = jsonWebKeySetHandler;
        this.issuerAcceptabilityHandler = issuerAcceptabilityHandler;
        this.permittedAlgs = new HashSet<String>(DEFAULT_PERMITTED_ALGS);
    }

    @Override
    public Set<String> getPermittedAlgorithms() {
        return ImmutableSet.copyOf(this.permittedAlgs);
    }

    @Override
    public JwtValidator setPermittedAlgorithms(Set<String> algorithms) throws NoSuchAlgorithmException {
        HashSet<String> copy = new HashSet<String>();
        for (String alg : algorithms) {
            if (!DEFAULT_PERMITTED_ALGS.contains(alg)) {
                throw new NoSuchAlgorithmException();
            }
            copy.add(alg);
        }
        this.permittedAlgs = copy;
        return this;
    }

    @Override
    public JwtValidator addPermittedAlgorithm(String algorithm) throws NoSuchAlgorithmException {
        if (!DEFAULT_PERMITTED_ALGS.contains(algorithm)) {
            throw new NoSuchAlgorithmException();
        }
        this.permittedAlgs.add(algorithm);
        return this;
    }

    @Override
    public JwtValidator setTimeLeeway(Duration timeLeeway) {
        this.timeLeewayMilliseconds = timeLeeway.toMillis();
        return this;
    }

    @Override
    public JwtValidator setRequireExp(boolean requireExp) {
        this.requireExp = requireExp;
        return this;
    }

    @Override
    public JwtValidator setRequireNbf(boolean requireNbf) {
        this.requireNbf = requireNbf;
        return this;
    }

    @Override
    public Future<Jwt> validateToken(String issuer, String token, List<String> requiredAudList, boolean ignoreRequiredAud) {
        Jwt jwt;
        try {
            jwt = Jwt.parseJws(token);
        }
        catch (Throwable ex2) {
            if (logger.isTraceEnabled()) {
                logger.error("Parse of JWT ({}) failed: ", (Object)token, (Object)ex2);
            } else {
                logger.error("Parse of JWT failed: ", ex2);
            }
            return Future.failedFuture((Throwable)new IllegalArgumentException("Parse of signed JWT failed", ex2));
        }
        try {
            this.validateAlgorithm(jwt.getAlgorithm());
            String kid = jwt.getKid();
            if (jwt.getPayloadSize() == 0) {
                logger.error("No payload claims found in JWT");
                return Future.failedFuture((Throwable)new IllegalArgumentException("Parse of signed JWT failed"));
            }
            return this.jsonWebKeySetHandler.findJwk(issuer, kid).onFailure(ex -> logger.warn("Failed to find JWK for {} ({}): ", new Object[]{kid, issuer, ex})).compose(jwk -> {
                try {
                    this.verify((JWK)jwk, jwt);
                    long now = System.currentTimeMillis();
                    this.validateIssuer(jwt, issuer);
                    this.validateNbf(jwt, now);
                    this.validateExp(jwt, now);
                    this.validateAud(jwt, requiredAudList, ignoreRequiredAud);
                    this.validateSub(jwt);
                    return Future.succeededFuture((Object)jwt);
                }
                catch (Throwable ex) {
                    logger.info("Validation of {} token failed: ", (Object)jwt.getAlgorithm(), (Object)ex);
                    return Future.failedFuture((Throwable)new IllegalArgumentException("Validation of " + jwt.getAlgorithm() + " signed JWT failed", ex));
                }
            });
        }
        catch (Throwable ex3) {
            logger.error("Failed to process token: ", ex3);
            return Future.failedFuture((Throwable)ex3);
        }
    }

    private void validateIssuer(Jwt jwt, String externalIssuer) {
        String tokenIssuer = jwt.getIssuer();
        if (Strings.isNullOrEmpty((String)tokenIssuer)) {
            throw new IllegalStateException("No issuer in token.");
        }
        if (!this.issuerAcceptabilityHandler.isAcceptable(tokenIssuer)) {
            throw new IllegalStateException("Issuer from token (" + tokenIssuer + ") is not acceptable.");
        }
        if (externalIssuer != null && !externalIssuer.equals(tokenIssuer)) {
            throw new IllegalStateException("Issuer from token (" + tokenIssuer + ") does not match expected issuer (" + externalIssuer + ").");
        }
    }

    private void verify(JWK jwk, Jwt jwt) throws IllegalArgumentException {
        if (Strings.isNullOrEmpty((String)jwt.getSignature())) {
            throw new IllegalStateException("No signature in token.");
        }
        Objects.requireNonNull(jwk, "JWK not set");
        if ("none".equals(jwk.getAlgorithm())) {
            throw new IllegalStateException("Algorithm \"none\" not allowed");
        }
        byte[] payloadInput = B64DECODER.decode(jwt.getSignature());
        byte[] signingInput = jwt.getSignatureBase().getBytes(StandardCharsets.UTF_8);
        try {
            JWS jws = new JWS(jwk);
            if (!jws.verify(payloadInput, signingInput)) {
                throw new IllegalArgumentException("Signature verification failed");
            }
        }
        catch (Throwable ex) {
            logger.warn("Signature verification failed: ", ex);
            throw new IllegalArgumentException("Signature verification failed", ex);
        }
    }

    private void validateSub(Jwt jwt) throws IllegalArgumentException {
        if (Strings.isNullOrEmpty((String)jwt.getSubject())) {
            throw new IllegalArgumentException("No subject specified in token");
        }
    }

    private void validateAud(Jwt jwt, List<String> requiredAudList, boolean ignoreRequiredAud) throws IllegalArgumentException {
        if (requiredAudList == null || !ignoreRequiredAud && requiredAudList.isEmpty()) {
            throw new IllegalStateException("Required audience not set");
        }
        if (jwt.getAudience() == null) {
            throw new IllegalArgumentException("Token does not include aud claim");
        }
        for (String aud : jwt.getAudience()) {
            for (String requiredAud : requiredAudList) {
                if (!requiredAud.equals(aud)) continue;
                return;
            }
        }
        if (!ignoreRequiredAud) {
            if (requiredAudList.size() == 1) {
                logger.warn("Required audience ({}) not found in token aud claim: {}", (Object)requiredAudList.get(0), jwt.getAudience());
            } else {
                logger.warn("None of the required audiences ({}) found in token aud claim: {}", requiredAudList, jwt.getAudience());
            }
            throw new IllegalArgumentException("Required audience not found in token");
        }
    }

    private void validateExp(Jwt jwt, long now) throws IllegalArgumentException {
        if (jwt.getExpiration() != null) {
            long targetMs = now - this.timeLeewayMilliseconds;
            if (1000L * jwt.getExpiration() < targetMs) {
                logger.warn("Token exp = {} ({}), now = {} ({}), target = {} ({})", new Object[]{jwt.getExpiration(), jwt.getExpirationLocalDateTime(), now, LocalDateTime.ofInstant(Instant.ofEpochMilli(now), ZoneOffset.UTC), targetMs, LocalDateTime.ofInstant(Instant.ofEpochMilli(targetMs), ZoneOffset.UTC)});
                throw new IllegalArgumentException("Token is not valid after " + String.valueOf(jwt.getExpirationLocalDateTime()));
            }
        } else if (this.requireExp) {
            throw new IllegalArgumentException("Token does not specify exp");
        }
    }

    private void validateNbf(Jwt jwt, long now) throws IllegalArgumentException {
        if (jwt.getNotBefore() != null) {
            long targetMs = now + this.timeLeewayMilliseconds;
            if (1000L * jwt.getNotBefore() > targetMs) {
                logger.warn("Token nbf = {} ({}), now = {} ({}), target = {} ({})", new Object[]{jwt.getNotBefore(), jwt.getNotBeforeLocalDateTime(), now, LocalDateTime.ofInstant(Instant.ofEpochMilli(now), ZoneOffset.UTC), targetMs, LocalDateTime.ofInstant(Instant.ofEpochMilli(targetMs), ZoneOffset.UTC)});
                throw new IllegalArgumentException("Token is not valid until " + String.valueOf(jwt.getNotBeforeLocalDateTime()));
            }
        } else if (this.requireNbf) {
            throw new IllegalArgumentException("Token does not specify nbf");
        }
    }

    private void validateAlgorithm(String algorithm) throws IllegalArgumentException {
        if (algorithm == null) {
            logger.warn("No signature algorithm in token.");
            throw new IllegalArgumentException("Parse of signed JWT failed");
        }
        if (!this.permittedAlgs.contains(algorithm)) {
            logger.warn("Failed to find algorithm \"{}\" in {}", (Object)algorithm, this.permittedAlgs);
            throw new IllegalArgumentException("Parse of signed JWT failed");
        }
    }
}

