package us.abstracta.jmeter.javadsl.codegeneration;

import java.lang.reflect.Method;
import java.util.List;
import org.apache.jmeter.testelement.TestElement;

/**
 * Abstracts common logic for usual scenario where MethodCallBuilders apply to only one type of
 * JMeter test element.
 * <p>
 * In most of the cases a {@link MethodCallBuilder} can simply extend this or {@link
 * SingleGuiClassCallBuilder} class. Scenarios where these classes would not be used are complex
 * ones which cover several test elements. Eg:
 * {@link us.abstracta.jmeter.javadsl.core.threadgroups.DslDefaultThreadGroup}
 * handles both JMeter default thread groups but also ultimate thread groups.
 * <p>
 * This class abstracts logic of checking if a test element MethodCall should be generated by this
 * builder or not, just comparing if the test element is of the type associated to the
 * MethodCallBuilder instance.
 *
 * @param <T> is the type of the JMeter test element instance used to check if the MethodCallBuilder
 *            applies to it or not.
 * @since 0.45
 */
public abstract class SingleTestElementCallBuilder<T extends TestElement> extends
    MethodCallBuilder {

  protected final Class<T> testElementClass;

  protected SingleTestElementCallBuilder(Class<T> testElementClass, List<Method> builderMethods) {
    super(builderMethods);
    this.testElementClass = testElementClass;
  }

  public boolean matches(MethodCallContext context) {
    return testElementClass == context.getTestElement().getClass();
  }

  @Override
  protected MethodCall buildMethodCall(MethodCallContext context) {
    return buildMethodCall(testElementClass.cast(context.getTestElement()), context);
  }

  /**
   * Builds a {@link MethodCall} for the given test element.
   * <p>
   * When this method is invoked, the test element has already been checked to be of the proper
   * type, so no further checking in that regard should be needed.
   *
   * @param testElement is the test element instance to build the MethodCall for.
   * @param context     is the context of the method call.
   * @return the generated MethodCall instance.
   */
  protected abstract MethodCall buildMethodCall(T testElement, MethodCallContext context);

}
