/*
========================================================================
SchemaCrawler
http://www.schemacrawler.com
Copyright (c) 2000-2018, Sualeh Fatehi <sualeh@hotmail.com>.
All rights reserved.
------------------------------------------------------------------------

SchemaCrawler is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

SchemaCrawler and the accompanying materials are made available under
the terms of the Eclipse Public License v1.0, GNU General Public License
v3 or GNU Lesser General Public License v3.

You may elect to redistribute this code under any of these licenses.

The Eclipse Public License is available at:
http://www.eclipse.org/legal/epl-v10.html

The GNU General Public License v3 and the GNU Lesser General Public
License v3 are available at:
http://www.gnu.org/licenses/

========================================================================
*/

package schemacrawler.crawl;


import static java.util.Objects.requireNonNull;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import schemacrawler.schema.CrawlInfo;
import schemacrawler.schema.DatabaseInfo;
import schemacrawler.schema.JdbcDriverInfo;
import schemacrawler.schema.OperatingSystemInfo;
import schemacrawler.schema.ProductVersion;
import schemacrawler.schema.SchemaCrawlerInfo;

/**
 * SchemaCrawler crawl information.
 *
 * @author Sualeh Fatehi sualeh@hotmail.com
 */
final class ImmutableCrawlInfo
  implements CrawlInfo
{

  private static final long serialVersionUID = 5982990326485881993L;

  private final ProductVersion schemaCrawlerInfo;
  private final ProductVersion jdbcDriverInfo;
  private final ProductVersion databaseInfo;
  private final ProductVersion osInfo;
  private final String title;
  private final LocalDateTime crawlTimestamp;

  ImmutableCrawlInfo(final SchemaCrawlerInfo schemaCrawlerInfo,
                     final JdbcDriverInfo jdbcDriverInfo,
                     final DatabaseInfo databaseInfo,
                     final OperatingSystemInfo osInfo,
                     final String title)
  {
    requireNonNull(schemaCrawlerInfo, "No SchemaCrawler information provided");
    this.schemaCrawlerInfo = new BaseProductVersion(schemaCrawlerInfo
      .getProductName(), schemaCrawlerInfo.getProductVersion());

    requireNonNull(jdbcDriverInfo, "No JDBC driver information provided");
    this.jdbcDriverInfo = new BaseProductVersion(jdbcDriverInfo
      .getProductName(), jdbcDriverInfo.getProductVersion());

    requireNonNull(databaseInfo, "No database information provided");
    this.databaseInfo = new BaseProductVersion(databaseInfo
      .getProductName(), databaseInfo.getProductVersion());

    requireNonNull(osInfo, "No database information provided");
    this.osInfo = new BaseProductVersion(osInfo.getProductName(),
                                         osInfo.getProductVersion());

    this.title = title;
    crawlTimestamp = LocalDateTime.now();
  }

  @Override
  public LocalDateTime getCrawlTimestamp()
  {
    return crawlTimestamp;
  }

  @Override
  public String getDatabaseInfo()
  {
    return databaseInfo.toString();
  }

  @Override
  public String getJdbcDriverInfo()
  {
    return jdbcDriverInfo.toString();
  }

  public String getOperatingSystemInfo()
  {
    return osInfo.toString();
  }

  @Override
  public String getSchemaCrawlerInfo()
  {
    return schemaCrawlerInfo.toString();
  }

  @Override
  public String getTitle()
  {
    return title;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString()
  {
    final StringBuilder info = new StringBuilder(1024);
    info.append("-- generated by: ").append(schemaCrawlerInfo)
      .append(System.lineSeparator());
    info
      .append("-- generated on: ").append(DateTimeFormatter
        .ofPattern("yyyy-MM-dd HH:mm:ss").format(getCrawlTimestamp()))
      .append(System.lineSeparator());
    info.append("-- database: ").append(databaseInfo)
      .append(System.lineSeparator());
    info.append("-- driver: ").append(jdbcDriverInfo)
      .append(System.lineSeparator());
    info.append("-- operating system: ").append(osInfo)
      .append(System.lineSeparator());
    return info.toString();
  }

}
