package us.ihmc.avatar.logging;

import us.ihmc.log.LogTools;
import us.ihmc.robotics.PlanarRegionFileTools;
import us.ihmc.robotics.geometry.PlanarRegionsList;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.Date;

public class PlanarRegionsListLogger
{
   private final String logName;
   private final int maxTicksToRecord;

   private boolean started = false;

   private OutputStream out = null;
   private BufferedWriter writer = null;

   private static final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd_HHmmss");
   private static final String logDirectory = System.getProperty("user.home") + File.separator + ".ihmc" + File.separator + "logs" + File.separator;

   public static String getLogDirectory()
   {
      return logDirectory;
   }

   public PlanarRegionsListLogger(String logName, int maxTicksToRecord)
   {
      this.logName = logName;
      this.maxTicksToRecord = maxTicksToRecord;
   }

   public void start()
   {
      File file = new File(logDirectory + dateFormat.format(new Date()) + "_" + this.getClass().getSimpleName() + ".prllog");
      try
      {
         out = new FileOutputStream(file);
         writer = new BufferedWriter(new OutputStreamWriter(out));
      }
      catch (IOException ex)
      {
         LogTools.error("Could not start PlanarRegionsListLogger!");
         LogTools.error(ex.getMessage());
         for (StackTraceElement e : ex.getStackTrace())
            LogTools.error(e.toString());

         return;
      }

      try
      {
         //Write header data
         writer.write("PlanarRegionsListLogger (PRLLOG) file\n");
         writer.write("Name: " + this.logName + "\n");
         writer.write("Generated on " + dateFormat.format(new Date()) + "\n");
         writer.write("Generated by " + new Throwable().getStackTrace()[1].getClassName() + "\n");
         writer.flush();
      }
      catch (IOException ex)
      {
         LogTools.error(ex.getMessage());
      }

      LogTools.info("Started PlanarRegionsListLogger");
      started = true;
   }

   public void update(long time, PlanarRegionsList list)
   {
      if (!started)
         return;

      try
      {
         writer.write("##\n");
         writer.write(time + "\n");
         writer.flush();

         PlanarRegionFileTools.exportPlanarRegionDataToStream(out, list);
      }
      catch (IOException ex)
      {
         LogTools.error("Failed to write to planar regions log");
         LogTools.error(ex.getMessage());
      }
   }

   @Override
   public void finalize()
   {
      try
      {
         if (out != null)
            out.close();
      }
      catch (IOException ignored)
      {
      }
   }
}
