package us.jakeabel.mpa.core;

import com.mongodb.client.MongoCollection;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.bson.types.ObjectId;
import us.jakeabel.mpa.core.api.Collection;
import us.jakeabel.mpa.util.MongoLoader;
import us.jakeabel.mpa.util.MongoUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

/**
 * Created by jakeabel on 5/29/16.
 *
 * Base class for all Mongo Repositories.
 *
 * The main DB Connection ties into this
 *
 *
 */
public class BaseRepo<T> implements IBaseRepo<T> {

    protected MongoCollection<Document> collection;
    private Class<T> genericType;

    public static <T extends DBModel> BaseRepo createBaseRepo(Class<T> clazz, String dbName) {
        return new BaseRepo<T>(clazz, dbName);
    }

    public Class<?> getGenericTypeOfthisClass() {
        Class<T> thisClass = null;
        Type type = getClass().getGenericSuperclass();
        if (type instanceof ParameterizedType) {
            ParameterizedType parameterizedType = (ParameterizedType) type;
            Type[] typeArguments = parameterizedType.getActualTypeArguments();
            thisClass = (Class<T>) typeArguments[0];
        }
        return thisClass;
    }


    public BaseRepo(Class<T> tClass, String dbName) {
        this.genericType = tClass;
        Collection colAnnotation = tClass.getAnnotation(Collection.class);
        this.collection = MongoLoader.getMongoDatabase(dbName).getCollection(colAnnotation.name());
    }

    @SuppressWarnings("unchecked")
    public BaseRepo() {
        this.genericType = (Class<T>)getGenericTypeOfthisClass();
    }


    @SuppressWarnings("unchecked")
    public BaseRepo(String collectionName) {
        this.collection = MongoLoader.getMongoDatabase().getCollection(collectionName);
        this.genericType = (Class<T>)getGenericTypeOfthisClass();
    }


    @SuppressWarnings("unchecked")
    public BaseRepo(String database, String collectionName) {
        this.collection = MongoLoader.getMongoDatabase(database).getCollection(collectionName);
        this.genericType = (Class<T>)getGenericTypeOfthisClass();
    }

    @SuppressWarnings("unchecked")
    public BaseRepo(String host, int port, String database, String collection) {
        this.collection = MongoLoader.getMongoDatabase(host, port, database).getCollection(collection);
        this.genericType = (Class<T>)getGenericTypeOfthisClass();
    }


    public BaseRepo(String host, int port, String dbName, Class<T> tClass) {
        this.genericType = tClass;
        Collection colAnnotation = tClass.getAnnotation(Collection.class);
        this.collection = MongoLoader.getMongoDatabase(host, port, dbName).getCollection(colAnnotation.name());
    }


    /**
     * gets the number of documents in a collection
     * @return long number of documents.
     */
    public long count() {
        return this.collection.count();
    }


    public long count(Document filter) {
        return this.collection.count(filter);
    }


    /**
     * Finds based on filter and puts them into a List.
     * Eases use of finding based on certain criteria.
     * @param filter Filter containing stuff to filter on.
     * @return List of items.
     */
    public List<T> find(Document filter) {
        List<T> items = new ArrayList<T>();
        collection.find(filter).forEach(new Consumer<Document>() {
            @Override
            public void accept(Document document) {
                items.add(MongoUtils.fromDocument(document, genericType));
            }
        });
        return items;
    }


    @Override
    public T findAndUpdate(Document filter, Document update, Class<T> clazz) {
        Document doc = collection.findOneAndUpdate(filter, update);
        return MongoUtils.fromDocument(doc, clazz);
    }

    public List<T> find() {
        List<T> items = new ArrayList<>();
        collection.find().forEach(new Consumer<Document>() {
            @Override
            public void accept(Document document) {
                items.add(MongoUtils.fromDocument(document, genericType));
            }
        });
        return items;
    }

    public List<T> find(int limit, int page) {
        List<T> items = new ArrayList<T>();
        collection.find().limit(limit).skip((page-1) * (limit)).forEach(new Consumer<Document>() {
            @Override
            public void accept(Document document) {
                items.add(MongoUtils.fromDocument(document, genericType));
            }
        });
        return items;
    }


    public T findLimit(Document filter, int limit) {
        List<T> items = new ArrayList<T>();
        collection.find().limit(limit).forEach(new Consumer<Document>() {
            @Override
            public void accept(Document document) {
                items.add(MongoUtils.fromDocument(document, genericType));
            }
        });
        if(items.size() == 0) {
            return null;
        }
        return items.get(0);
    }

    /**
     * checks to see if the item exists in the database.
     * @param filter Item to search for.
     * @return exists
     */
    @Override
    public boolean exists(Document filter) {
        return count(filter) > 0;
    }


    /**
     * Finds based on the Mongo _id.
     * @param _id String id.
     * @return Single Object.
     */
    @Override
    public T find(String _id) {
        return findFirstItem(new Document("_id", new ObjectId(_id)));
    }



    public T findFirst(Document doc) {
        return findFirstItem(doc);
    }



    private T findFirstItem(Document filter) {
        Document first = collection.find(filter).first();
        if(first == null) return null;
        return MongoUtils.fromDocument(first, genericType);
    }


    public T insert(T obj) {
        Document doc = MongoUtils.toDocument(obj);
        collection.insertOne(doc);
        return findFirstItem(MongoUtils.toDocument(obj));
    }


    @Override
    public DeleteResult delete(T obj) {
        return collection.deleteOne(new Document("_id", new ObjectId(((DBModel)obj).getId())));
    }

    @Override
    public DeleteResult delete(String id) {
        return collection.deleteOne(new Document("_id", new ObjectId(id)));
    }

    @Override
    public UpdateResult update(T obj) {
        return collection.updateOne(new Document("_id", new ObjectId(((DBModel)obj).getId())),
                new Document("$set", MongoUtils.toDocument(obj)));

    }

    public MongoCollection<Document> getCollection() {
        return this.collection;
    }
}
