package us.jakeabel.mpa.core;

import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.reflections.Reflections;
import us.jakeabel.mpa.core.api.Collection;

import java.io.IOException;
import java.util.*;

/**
 * Created by jake on 10/15/16.
 *
 *
 * Delegates the operations over to the map of base repos.
 *
 */

@SuppressWarnings("unchecked")
public class BaseConnection implements IConnection {



    protected Properties properties = null;

    protected Map<Class<?>, BaseRepo> repos = new HashMap<>();


    /**
     * Example connection string is: localhost:27017/testdb
     *          somecompany.com:27017/testdb is okay too.
     * @param connectionString Raw connection string.
     * @throws IOException
     */
    BaseConnection(String connectionString) throws IOException {
        String host = null;
        int port = -1;
        String dbName = null;
        if(connectionString.matches(".*:\\d+/.*")) {
            host = connectionString.replaceAll(":.*", "");
            String portStr = connectionString.replaceAll("/.*", "").replaceAll(".*:", "");
            port = Integer.parseInt(portStr);
            dbName = connectionString.replaceAll(".*/", "");
        }

        if(host == null && port == -1 || dbName == null) {
            throw new IOException("Invalid data exception");
        }

        ClassPathResource cpr = new ClassPathResource("connection.properties");
        this.properties = new Properties();
        this.properties.load(cpr.getInputStream());

        String basePackage = this.properties.getProperty("basePackage");
        System.out.println("Scanning package: " + basePackage);

        Reflections reflections = new Reflections(basePackage);
        Set<Class<?>> clazzes = reflections.getTypesAnnotatedWith(Collection.class);

        for(Class<?> c : clazzes) {
            System.out.println(c.getSimpleName());
            Class claz = null;
            try {
                claz = c.newInstance().getClass();
            } catch (Exception ex1) {
                ex1.printStackTrace();
            }
            if (claz != null) {
                this.repos.put(c, BaseRepo.createBaseRepo(claz, host, port, dbName));
            }
        }
    }


    BaseConnection() throws IOException {
        // load the data from the properties file.
        ClassPathResource cpr = new ClassPathResource("connection.properties");
        this.properties = new Properties();
        this.properties.load(cpr.getInputStream());

        String basePackage = this.properties.getProperty("basePackage");
        System.out.println("Scanning package: " + basePackage);

        String dbName = this.properties.getProperty("defaultdb");
        Reflections reflections = new Reflections(basePackage);
        Set<Class<?>> clazzes = reflections.getTypesAnnotatedWith(Collection.class);

        for(Class<?> c : clazzes) {
            System.out.println(c.getSimpleName());
            Class claz = null;
            try {
                claz = c.newInstance().getClass();
            } catch (Exception ex1) {
                ex1.printStackTrace();
            }
            if (claz != null) {
                this.repos.put(c, BaseRepo.createBaseRepo(claz, dbName));
            }
        }
    }

    @Override
    public <T> List<T> find(Document filter, Class<T> clazz) {
        return repos.get(clazz).find(filter);
    }

    @Override
    public <T> List<T> find(Class<T> clazz) {
        return repos.get(clazz).find();
    }

    @Override
    public <T> T find(String id, Class<T> clazz) {
        List<T> list = repos.get(clazz).find(new Document("_id", new ObjectId(id)));
        if(list.size() == 0) {
            return null;
        }
        return list.get(0);
    }

    @Override
    public <T> List<T> findAndUpdate(Document filter, Document update, Class<T> clazz) {
        return repos.get(clazz).find(filter);
    }

    @Override
    public <T> T findFirstAndUpdate(Document filter, Document update, Class<T> clazz) {
        return (T)repos.get(clazz).findAndUpdate(filter, update, clazz);
    }

    @Override
    public <T> T findFirst(Document filter, Class<T> clazz) {
        List<T> list = repos.get(clazz).find(filter);
        if(list.size() == 0) {
            return null;
        }
        return list.get(0);
    }

    @Override
    public <T> boolean exists(Document filter, Class<T> clazz) {
        return count(filter, clazz) > 0;
    }

    @Override
    public long count(Class<? extends DBModel> clazz) {
        return repos.get(clazz).count();
    }

    @Override
    public <T> long count(Document filter, Class<T> clazz) {
        return repos.get(clazz).count(filter);
    }

    @Override
    public <T> T insert(T obj) {
        return (T)repos.get(obj.getClass()).insert(obj);
    }

    @Override
    public <T> T create(T obj) {
        return (T) repos.get(obj.getClass()).insert(obj);
    }

    @Override
    public DeleteResult delete(DBModel obj) {
        return repos.get(obj.getClass()).delete(obj);
    }

    @Override
    public DeleteResult delete(String id, Class<? extends DBModel> clazz) {
        return repos.get(clazz).delete(id);

    }

    @Override
    public UpdateResult update(DBModel obj) {
        return repos.get(obj.getClass()).update(obj);
    }
}
