package us.jakeabel.mpa.util;


import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * Created by jake on 4/19/16.
 *
 *
 * Date Utilities
 *
 */
public class DateUtils {


    public static String toIso8601(Date date) {
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
        return df.format(date);
    }


    /**
     * Parses ISO Date to Date Object
     * @param dateStr ISO 8601 string
     * @return parsed date
     */
    public static Date fromIso8601(String dateStr) {
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
        Date date = new Date();
        try {
            date = df.parse(dateStr);
        } catch (ParseException pe) {
            pe.printStackTrace();
        }
        return date;
    }


    /**
     * Gets a month integer from a month string.
     * @param monthStr Month string Jan or January etc...
     * @return Integer of the month
     */
    public static int getMonthFromString(String monthStr) {
        int month = Calendar.JANUARY;
        if(monthStr.matches("(?i).*((january)|(jan)).*")) {
            return Calendar.JANUARY;
        }
        if(monthStr.matches("(?i).*((february)|(feb)).*")) {
            return Calendar.FEBRUARY;
        }
        if(monthStr.matches("(?i).*((march)|(mar)).*")) {
            return Calendar.MARCH;
        }
        if(monthStr.matches("(?i).*((april)|(apr)).*")) {
            return Calendar.APRIL;
        }
        if(monthStr.matches("(?i).*(may).*")) {
            return Calendar.MAY;
        }
        if(monthStr.matches("(?i).*((june)|(jun)).*")) {
            return Calendar.JUNE;
        }
        if(monthStr.matches("(?i).*((july)|(jul)).*")) {
            return Calendar.JULY;
        }
        if(monthStr.matches("(?i).*((august)|(aug)).*")) {
            return Calendar.AUGUST;
        }
        if(monthStr.matches("(?i).*((september)|(sep)|(sept)).*")) {
            return Calendar.SEPTEMBER;
        }
        if(monthStr.matches("(?i).*((october)|(oct)).*")) {
            return Calendar.OCTOBER;
        }
        if(monthStr.matches("(?i).*((november)|(nov)).*")) {
            return Calendar.NOVEMBER;
        }
        if(monthStr.matches("(?i).*((december)|(dec)).*")) {
            return Calendar.DECEMBER;
        }
        return Calendar.JANUARY;
    }


    /**
     * Parses date from an unknown format to a Date Object.
     * @param dateString Raw date string
     * @return Calendar
     */
    public static Calendar parseCalendar(String dateString) {
        if(dateString.matches(".*/.*") && dateString.matches(".*:.*")) {
            String[] parts = dateString.split("\\s");
            int month = 0;
            int day = 0;
            int year = 0;
            int hour = 0;
            int minute = 0;
            for(String part : parts) {
                if(part.matches("\\d{1,2}/\\d{1,2}/\\d{2,4}")) {
                    String[] dateParts = part.split("/");
                    if(dateParts.length == 3) {
                        month = Integer.parseInt(dateParts[0]);
                        day = Integer.parseInt(dateParts[1]);
                        year = Integer.parseInt(dateParts[2]);
                    }
                }
                else if(part.matches("\\d{1,2}:\\d{2}")) {
                    String[] dateParts = part.split(":");
                    if(dateParts.length == 2) {
                        hour = Integer.parseInt(dateParts[0]);
                        minute = Integer.parseInt(dateParts[1]);
                    }
                }
                else if(part.matches("(AM)|(PM)")) {
                    if(part.matches("(?i)pm")) hour += 12;
                }
            }
            return new GregorianCalendar(year, month, day, hour, minute);
        }
        else if(!dateString.contains(":")) {
            dateString = dateString.replaceAll(RegexConstants.DAY_OF_WEEK_REGEX, " ").replaceAll("\\s+", " ").trim();
            dateString = dateString.replaceAll("^,", "").trim();

            int month = getMonthFromString(dateString);
            dateString = dateString.replaceAll(RegexConstants.MONTH_REGEX, " ").replaceAll("\\s+", " ").trim();
            int dayOfMonth = 0;
            int year = 0;
            String[] parts = dateString.split(",");
            for(String p : parts) {
                p = p.trim();
                if(p.matches("\\d{1,2}")) {
                    dayOfMonth = Integer.parseInt(p);
                }
                else if(p.matches("\\d{4}")) {
                    year = Integer.parseInt(p);
                }
            }
            return new GregorianCalendar(year, month, dayOfMonth);
        }
        else if(dateString.matches(".*" + RegexConstants.MONTH_REGEX + ".*")) {
            dateString = dateString.replaceAll(RegexConstants.DAY_OF_WEEK_REGEX, " ");
            dateString = dateString.trim().replaceAll("^,", "").trim();

            int month = getMonthFromString(dateString);
            dateString = dateString.replaceAll(RegexConstants.MONTH_REGEX, " ").replaceAll("\\s+", " ");
            String[] parts = dateString.split("((\\s)|(,))");
            int dayOfMonth = 0;
            int year = 0;
            int hour = 0;
            int minute = 0;
            int amOrPm = 0;
            for(String part : parts) {
                if(part.equals("")) continue;
                part = part.trim();
                if(part.matches("[\\d]{4}")) { // year
                    year = Integer.parseInt(part);
                }
                else if(part.matches("\\d{1,2}")){
                    dayOfMonth = Integer.parseInt(part);
                }
                else if(part.matches("\\d{1,2}:\\d{2}")){
                    String[] timeParts = part.split(":");
                    hour = Integer.parseInt(timeParts[0]);
                    minute = Integer.parseInt(timeParts[1]);
                }
                else if(part.matches("(?i)am")){
                    amOrPm = Calendar.AM;
                }
                else if(part.matches("(?i)pm")) {
                    amOrPm = Calendar.PM;
                }
            }
            if(amOrPm == 1) {
                hour += 12;
            }
            return new GregorianCalendar(year, month, dayOfMonth, hour, minute);
        }


        return GregorianCalendar.getInstance();
    }


    /**
     * parses date string to a date format
     * @param dateString raw date string
     * @return Date
     */
    public static Date parseDate(String dateString) {
        return parseCalendar(dateString).getTime();
    }


    /**
     * Parses a raw date string to an iso 8601 date string
     * @param dateString Raw date string.
     * @return ISO 8601 date string
     */
    public static String parseDateToIso(String dateString) {
        return toIso8601(parseDate(dateString));
    }


}
