/*
 * Decompiled with CFR 0.152.
 */
package alpine.security.crypto;

import alpine.Config;
import alpine.common.logging.Logger;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

public final class KeyManager {
    private static final Logger LOGGER = Logger.getLogger(KeyManager.class);
    private static final KeyManager INSTANCE = new KeyManager();
    private KeyPair keyPair;
    private SecretKey secretKey;

    private KeyManager() {
        this.initialize();
    }

    public static KeyManager getInstance() {
        return INSTANCE;
    }

    private void initialize() {
        this.createKeysIfNotExist();
        if (this.keyPair == null) {
            try {
                this.loadKeyPair();
            }
            catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
                LOGGER.error("An error occurred loading key pair");
                LOGGER.error(e.getMessage());
            }
        }
        if (this.secretKey == null) {
            try {
                if (this.secretKeyHasOldFormat()) {
                    this.loadSecretKey();
                } else {
                    this.loadEncodedSecretKey();
                }
            }
            catch (IOException | ClassNotFoundException e) {
                LOGGER.error("An error occurred loading secret key");
                LOGGER.error(e.getMessage());
            }
        }
    }

    private void createKeysIfNotExist() {
        if (!this.keyPairExists()) {
            try {
                KeyPair keyPair = this.generateKeyPair();
                this.save(keyPair);
            }
            catch (NoSuchAlgorithmException e) {
                LOGGER.error("An error occurred generating new keypair");
                LOGGER.error(e.getMessage());
            }
            catch (IOException e) {
                LOGGER.error("An error occurred saving newly generated keypair");
                LOGGER.error(e.getMessage());
            }
        }
        if (!this.secretKeyExists()) {
            try {
                SecretKey secretKey = this.generateSecretKey();
                this.saveEncoded(secretKey);
            }
            catch (NoSuchAlgorithmException e) {
                LOGGER.error("An error occurred generating new secret key");
                LOGGER.error(e.getMessage());
            }
            catch (IOException e) {
                LOGGER.error("An error occurred saving newly generated secret key");
                LOGGER.error(e.getMessage());
            }
        }
    }

    public KeyPair generateKeyPair() throws NoSuchAlgorithmException {
        LOGGER.info("Generating new key pair");
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
        SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
        keyGen.initialize(4096, random);
        this.keyPair = keyGen.generateKeyPair();
        return this.keyPair;
    }

    public SecretKey generateSecretKey() throws NoSuchAlgorithmException {
        KeyGenerator keyGen = KeyGenerator.getInstance("AES");
        SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
        keyGen.init(256, random);
        this.secretKey = keyGen.generateKey();
        return this.secretKey;
    }

    private File getKeyPath(KeyType keyType) {
        String secretKeyPath;
        if (keyType == KeyType.SECRET && (secretKeyPath = Config.getInstance().getProperty((Config.Key)Config.AlpineKey.SECRET_KEY_PATH)) != null) {
            return Paths.get(secretKeyPath, new String[0]).toFile();
        }
        return new File(Config.getInstance().getDataDirectorty() + File.separator + "keys" + File.separator + keyType.name().toLowerCase() + ".key");
    }

    private File getKeyPath(Key key) {
        KeyType keyType = null;
        if (key instanceof PrivateKey) {
            keyType = KeyType.PRIVATE;
        } else if (key instanceof PublicKey) {
            keyType = KeyType.PUBLIC;
        } else if (key instanceof SecretKey) {
            keyType = KeyType.SECRET;
        }
        return this.getKeyPath(keyType);
    }

    public void save(KeyPair keyPair) throws IOException {
        LOGGER.info("Saving key pair");
        PrivateKey privateKey = keyPair.getPrivate();
        PublicKey publicKey = keyPair.getPublic();
        File publicKeyFile = this.getKeyPath(publicKey);
        publicKeyFile.getParentFile().mkdirs();
        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(publicKey.getEncoded());
        try (OutputStream fos = Files.newOutputStream(publicKeyFile.toPath(), new OpenOption[0]);){
            fos.write(x509EncodedKeySpec.getEncoded());
        }
        File privateKeyFile = this.getKeyPath(privateKey);
        privateKeyFile.getParentFile().mkdirs();
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(privateKey.getEncoded());
        try (OutputStream fos = Files.newOutputStream(privateKeyFile.toPath(), new OpenOption[0]);){
            fos.write(pkcs8EncodedKeySpec.getEncoded());
        }
    }

    @Deprecated(forRemoval=true)
    public void save(SecretKey key) throws IOException {
        File keyFile = this.getKeyPath(key);
        keyFile.getParentFile().mkdirs();
        try (OutputStream fos = Files.newOutputStream(keyFile.toPath(), new OpenOption[0]);
             ObjectOutputStream oout = new ObjectOutputStream(fos);){
            oout.writeObject(key);
        }
    }

    public void saveEncoded(SecretKey key) throws IOException {
        File keyFile = this.getKeyPath(key);
        keyFile.getParentFile().mkdirs();
        try (OutputStream fos = Files.newOutputStream(keyFile.toPath(), new OpenOption[0]);){
            fos.write(key.getEncoded());
        }
    }

    private KeyPair loadKeyPair() throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] encodedPublicKey;
        byte[] encodedPrivateKey;
        File filePrivateKey = this.getKeyPath(KeyType.PRIVATE);
        File filePublicKey = this.getKeyPath(KeyType.PUBLIC);
        try (InputStream pvtfis = Files.newInputStream(filePrivateKey.toPath(), new OpenOption[0]);
             InputStream pubfis = Files.newInputStream(filePublicKey.toPath(), new OpenOption[0]);){
            encodedPrivateKey = new byte[(int)filePrivateKey.length()];
            pvtfis.read(encodedPrivateKey);
            encodedPublicKey = new byte[(int)filePublicKey.length()];
            pubfis.read(encodedPublicKey);
        }
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(encodedPublicKey);
        PublicKey publicKey = keyFactory.generatePublic(publicKeySpec);
        PKCS8EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(encodedPrivateKey);
        PrivateKey privateKey = keyFactory.generatePrivate(privateKeySpec);
        this.keyPair = new KeyPair(publicKey, privateKey);
        return this.keyPair;
    }

    @Deprecated(forRemoval=true)
    SecretKey loadSecretKey() throws IOException, ClassNotFoundException {
        SecretKey key;
        File file = this.getKeyPath(KeyType.SECRET);
        try (InputStream fis = Files.newInputStream(file.toPath(), new OpenOption[0]);
             ObjectInputStream ois = new ObjectInputStream(fis);){
            key = (SecretKey)ois.readObject();
        }
        this.secretKey = key;
        return this.secretKey;
    }

    SecretKey loadEncodedSecretKey() throws IOException {
        File file = this.getKeyPath(KeyType.SECRET);
        try (InputStream fis = Files.newInputStream(file.toPath(), new OpenOption[0]);){
            byte[] encodedKey = fis.readAllBytes();
            this.secretKey = new SecretKeySpec(encodedKey, 0, encodedKey.length, "AES");
            SecretKeySpec secretKeySpec = this.secretKey;
            return secretKeySpec;
        }
    }

    public boolean keyPairExists() {
        return this.getKeyPath(KeyType.PUBLIC).exists() && this.getKeyPath(KeyType.PRIVATE).exists();
    }

    public boolean secretKeyExists() {
        return this.getKeyPath(KeyType.SECRET).exists();
    }

    boolean secretKeyHasOldFormat() throws IOException {
        try (InputStream fis = Files.newInputStream(this.getKeyPath(KeyType.SECRET).toPath(), new OpenOption[0]);){
            boolean bl = ByteBuffer.wrap(fis.readNBytes(2)).getShort() == -21267;
            return bl;
        }
    }

    public KeyPair getKeyPair() {
        return this.keyPair;
    }

    public PublicKey getPublicKey() {
        return this.keyPair != null ? this.keyPair.getPublic() : null;
    }

    public PrivateKey getPrivateKey() {
        return this.keyPair != null ? this.keyPair.getPrivate() : null;
    }

    public SecretKey getSecretKey() {
        return this.secretKey;
    }

    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    static enum KeyType {
        PRIVATE,
        PUBLIC,
        SECRET;

    }
}

