/*
 * Decompiled with CFR 0.152.
 */
package alpine.server.filters;

import alpine.common.logging.Logger;
import alpine.event.framework.LoggableUncaughtExceptionHandler;
import alpine.model.ApiKey;
import alpine.persistence.AlpineQueryManager;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.jdo.PersistenceManager;
import javax.jdo.PersistenceManagerFactory;
import javax.jdo.datastore.JDOConnection;
import javax.ws.rs.ext.Provider;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.glassfish.jersey.server.monitoring.ApplicationEvent;
import org.glassfish.jersey.server.monitoring.ApplicationEventListener;
import org.glassfish.jersey.server.monitoring.RequestEvent;
import org.glassfish.jersey.server.monitoring.RequestEventListener;

@Provider
public class ApiKeyUsageTracker
implements ApplicationEventListener {
    private static final Logger LOGGER = Logger.getLogger(ApiKeyUsageTracker.class);
    private static final BlockingQueue<ApiKeyUsedEvent> EVENT_QUEUE = new ArrayBlockingQueue<ApiKeyUsedEvent>(10000);
    private final ScheduledExecutorService flushExecutor;
    private final Lock flushLock;

    public ApiKeyUsageTracker() {
        BasicThreadFactory threadFactory = new BasicThreadFactory.Builder().uncaughtExceptionHandler((Thread.UncaughtExceptionHandler)new LoggableUncaughtExceptionHandler()).namingPattern("Alpine-ApiKeyUsageTracker-%d").build();
        this.flushExecutor = Executors.newSingleThreadScheduledExecutor((ThreadFactory)threadFactory);
        this.flushLock = new ReentrantLock();
    }

    public void onEvent(ApplicationEvent event) {
        switch (event.getType()) {
            case INITIALIZATION_FINISHED: {
                this.flushExecutor.scheduleAtFixedRate(this::flush, 5L, 30L, TimeUnit.SECONDS);
                break;
            }
            case DESTROY_FINISHED: {
                this.flushExecutor.shutdown();
                try {
                    boolean terminated = this.flushExecutor.awaitTermination(5L, TimeUnit.SECONDS);
                    if (!terminated) {
                        LOGGER.warn("Flush executor did not terminate on time (waited for 5s); Remaining events in the queue: %d".formatted(EVENT_QUEUE.size()));
                    }
                }
                catch (InterruptedException e) {
                    LOGGER.warn("Interrupted while waiting for pending flush tasks to complete");
                    Thread.currentThread().interrupt();
                }
                this.flush();
            }
        }
    }

    public RequestEventListener onRequest(RequestEvent requestEvent) {
        return null;
    }

    static void onApiKeyUsed(ApiKey apiKey) {
        ApiKeyUsedEvent event = new ApiKeyUsedEvent(apiKey.getId(), Instant.now().toEpochMilli());
        if (!EVENT_QUEUE.offer(event)) {
            LOGGER.debug("Usage of API key %s can not be tracked because the event queue is already saturated".formatted(apiKey.getMaskedKey()));
        }
    }

    private void flush() {
        try {
            this.flushLock.lock();
            if (EVENT_QUEUE.isEmpty()) {
                return;
            }
            HashMap<Long, Long> lastUsedByKeyId = new HashMap<Long, Long>();
            while (EVENT_QUEUE.peek() != null) {
                ApiKeyUsedEvent event = (ApiKeyUsedEvent)EVENT_QUEUE.poll();
                lastUsedByKeyId.compute(event.keyId(), (ignored, prev) -> {
                    if (prev == null) {
                        return event.timestamp();
                    }
                    return Math.max(prev, event.timestamp());
                });
            }
            LOGGER.debug("Updating last used timestamps for %d API keys".formatted(lastUsedByKeyId.size()));
            this.updateLastUsed(lastUsedByKeyId);
        }
        catch (Exception e) {
            LOGGER.error("Failed to update last used timestamps of API keys", (Throwable)e);
        }
        finally {
            this.flushLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateLastUsed(Map<Long, Long> lastUsedByKeyId) throws SQLException {
        try (AlpineQueryManager qm = new AlpineQueryManager();){
            PersistenceManager pm = qm.getPersistenceManager();
            JDOConnection jdoConnection = pm.getDataStoreConnection();
            Connection connection = (Connection)jdoConnection.getNativeConnection();
            try (PreparedStatement ps = connection.prepareStatement("UPDATE \"APIKEY\" SET \"LAST_USED\" = ?\nWHERE \"ID\" = ? AND (\"LAST_USED\" IS NULL OR \"LAST_USED\" < ?)\n");){
                for (Map.Entry<Long, Long> entry : lastUsedByKeyId.entrySet()) {
                    Timestamp lastUsed = new Timestamp(entry.getValue());
                    ps.setTimestamp(1, lastUsed);
                    ps.setLong(2, entry.getKey());
                    ps.setTimestamp(3, lastUsed);
                    ps.addBatch();
                }
                ps.executeBatch();
            }
            finally {
                jdoConnection.close();
            }
            PersistenceManagerFactory pmf = pm.getPersistenceManagerFactory();
            pmf.getDataStoreCache().evictAll(false, ApiKey.class);
        }
    }

    private record ApiKeyUsedEvent(long keyId, long timestamp) {
    }
}

