/*
 * Decompiled with CFR 0.152.
 */
package us.springett.nistdatamirror;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Authenticator;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.PasswordAuthentication;
import java.net.Proxy;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import us.springett.nistdatamirror.MetaProperties;
import us.springett.nistdatamirror.MirrorException;

public class NistDataMirror {
    private static final int EXIT_CODE_DOWNLOAD_FAILED = 1;
    private static final int EXIT_CODE_WRONG_INVOCATION = 2;
    private static final String CVE_JSON_11_MODIFIED_URL = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-modified.json.gz";
    private static final String CVE_JSON_11_RECENT_URL = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-recent.json.gz";
    private static final String CVE_JSON_11_BASE_URL = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-%d.json.gz";
    private static final String CVE_MODIFIED_11_META = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-modified.meta";
    private static final String CVE_RECENT_11_META = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-recent.meta";
    private static final String CVE_BASE_11_META = "https://nvd.nist.gov/feeds/json/cve/1.1/nvdcve-1.1-%d.meta";
    private static final Map<String, Map<String, String>> versionToFilenameMaps = new HashMap<String, Map<String, String>>();
    private static final int START_YEAR = 2002;
    private static final int END_YEAR = Calendar.getInstance().get(1);
    private final File outputDir;
    private boolean downloadFailed = false;
    private final Proxy proxy;
    private static final char[] HEX_ARRAY;

    public static void main(String[] args) {
        if (args.length != 1) {
            System.out.println("Usage: java NistDataMirror outputDir");
            System.exit(2);
            return;
        }
        NistDataMirror nvd = new NistDataMirror(args[0]);
        nvd.mirror("1.1");
        if (nvd.downloadFailed) {
            System.exit(1);
        }
    }

    public NistDataMirror(String outputDirPath) {
        boolean mkdirOk;
        this.outputDir = new File(outputDirPath);
        if (!this.outputDir.exists() && !(mkdirOk = this.outputDir.mkdirs())) {
            System.out.println("Could not create " + this.outputDir.getAbsolutePath() + " even though it did not exist.");
        }
        this.proxy = this.initProxy();
    }

    private Proxy initProxy() {
        String proxyHost = System.getProperty("http.proxyHost");
        String proxyPort = System.getProperty("http.proxyPort");
        if (proxyHost != null && !proxyHost.trim().isEmpty() && proxyPort != null && !proxyPort.trim().isEmpty()) {
            System.out.println("Using proxy " + proxyHost + ":" + proxyPort);
            final String proxyUser = System.getProperty("http.proxyUser");
            final String proxyPassword = System.getProperty("http.proxyPassword");
            if (proxyUser != null && !proxyUser.trim().isEmpty() && proxyPassword != null && !proxyPassword.trim().isEmpty()) {
                System.out.println("Using proxy user " + proxyUser + ":" + proxyPassword);
                Authenticator authenticator = new Authenticator(){

                    @Override
                    public PasswordAuthentication getPasswordAuthentication() {
                        return new PasswordAuthentication(proxyUser, proxyPassword.toCharArray());
                    }
                };
                Authenticator.setDefault(authenticator);
            }
            return new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, Integer.parseInt(proxyPort)));
        }
        return Proxy.NO_PROXY;
    }

    public void mirror(String version) {
        try {
            long seconds;
            long hours;
            Date currentDate = new Date();
            System.out.println("Downloading files at " + currentDate);
            MetaProperties before = this.readLocalMetaForURL(versionToFilenameMaps.get(version).get("cveModifiedMeta"));
            if (before != null && (hours = (seconds = ZonedDateTime.now().toEpochSecond() - before.getLastModifiedDate()) / 60L / 60L) < 2L) {
                System.out.println("Using local NVD cache as last update was within two hours");
                return;
            }
            this.doDownload(versionToFilenameMaps.get(version).get("cveModifiedMeta"));
            MetaProperties after = this.readLocalMetaForURL(versionToFilenameMaps.get(version).get("cveModifiedMeta"));
            if (before == null || after.getLastModifiedDate() > before.getLastModifiedDate()) {
                this.doDownload(versionToFilenameMaps.get(version).get("cveJsonModifiedUrl"));
            }
            before = this.readLocalMetaForURL(versionToFilenameMaps.get(version).get("cveRecentMeta"));
            this.doDownload(versionToFilenameMaps.get(version).get("cveRecentMeta"));
            after = this.readLocalMetaForURL(versionToFilenameMaps.get(version).get("cveRecentMeta"));
            if (before == null || after.getLastModifiedDate() > before.getLastModifiedDate()) {
                this.doDownload(versionToFilenameMaps.get(version).get("cveJsonRecentUrl"));
            }
            for (int year = 2002; year <= END_YEAR; ++year) {
                this.downloadVersionForYear(version, year);
                Boolean valid = this.validCheck(year);
                System.out.println("File " + year + " is valid.");
                if (!Boolean.FALSE.equals(valid)) continue;
                for (int i = 0; i < 2; ++i) {
                    this.downloadVersionForYear(version, year);
                    Boolean valid2 = this.validCheck(year);
                    if (!Boolean.TRUE.equals(valid2)) continue;
                    System.out.println("File " + year + " is valid.");
                    break;
                }
                System.out.println("The File " + year + " is corrupted");
            }
        }
        catch (MirrorException ex) {
            this.downloadFailed = true;
            System.err.println("Error mirroring the NVD CVE data");
            ex.printStackTrace(System.err);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void downloadVersionForYear(String version, int year) throws MirrorException {
        String cveBaseMetaUrl = versionToFilenameMaps.get(version).get("cveBaseMeta").replace("%d", String.valueOf(year));
        MetaProperties before = this.readLocalMetaForURL(cveBaseMetaUrl);
        this.doDownload(cveBaseMetaUrl);
        MetaProperties after = this.readLocalMetaForURL(cveBaseMetaUrl);
        if (before == null || after.getLastModifiedDate() > before.getLastModifiedDate()) {
            String cveJsonBaseUrl = versionToFilenameMaps.get(version).get("cveJsonBaseUrl").replace("%d", String.valueOf(year));
            this.doDownload(cveJsonBaseUrl);
        }
    }

    private MetaProperties readLocalMetaForURL(String metaUrl) throws MirrorException {
        URL url;
        try {
            url = new URL(metaUrl);
        }
        catch (MalformedURLException ex) {
            throw new MirrorException("Invalid url: " + metaUrl, ex);
        }
        MetaProperties meta = null;
        String filename = url.getFile();
        filename = filename.substring(filename.lastIndexOf(47) + 1);
        File file = new File(this.outputDir, filename).getAbsoluteFile();
        if (file.isFile()) {
            meta = new MetaProperties(file);
        }
        return meta;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doDownload(String nvdUrl) throws MirrorException {
        URL url;
        try {
            url = new URL(nvdUrl);
        }
        catch (MalformedURLException ex) {
            throw new MirrorException("Invalid url: " + nvdUrl, ex);
        }
        BufferedInputStream bis = null;
        BufferedOutputStream bos = null;
        File file = null;
        boolean success = false;
        try {
            int i;
            String filename = url.getFile();
            filename = filename.substring(filename.lastIndexOf(47) + 1);
            file = new File(this.outputDir, filename).getAbsoluteFile();
            URLConnection connection = url.openConnection(this.proxy);
            System.out.println("Downloading " + url.toExternalForm());
            bis = new BufferedInputStream(connection.getInputStream());
            file = new File(this.outputDir, filename);
            bos = new BufferedOutputStream(new FileOutputStream(file));
            while ((i = bis.read()) != -1) {
                bos.write(i);
            }
            success = true;
            this.close(bis);
            this.close(bos);
        }
        catch (IOException e) {
            System.out.println("Download failed : " + e.getLocalizedMessage());
            this.downloadFailed = true;
        }
        finally {
            this.close(bis);
            this.close(bos);
        }
        if (file != null && success) {
            System.out.println("Download succeeded " + file.getName());
            if (file.getName().endsWith(".gz")) {
                this.uncompress(file);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void uncompress(File file) {
        byte[] buffer = new byte[1024];
        GZIPInputStream gzis = null;
        BufferedOutputStream out = null;
        try {
            int len;
            File outputFile = new File(file.getAbsolutePath().replaceAll(".gz", ""));
            gzis = new GZIPInputStream(new BufferedInputStream(new FileInputStream(file)));
            out = new BufferedOutputStream(new FileOutputStream(outputFile));
            while ((len = ((InputStream)gzis).read(buffer)) > 0) {
                ((OutputStream)out).write(buffer, 0, len);
            }
            System.out.println("Uncompressed " + outputFile.getName());
            this.close(gzis);
            this.close(out);
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
        finally {
            this.close(gzis);
            this.close(out);
        }
    }

    private void close(Closeable object) {
        if (object != null) {
            try {
                object.close();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private Boolean validCheck(int year) {
        try {
            Path metaFilePath = Paths.get(String.valueOf(this.outputDir), "nvdcve-1.1-" + year + ".meta");
            int n = 4;
            String hashLine = Files.readAllLines(Paths.get(String.valueOf(metaFilePath), new String[0])).get(n);
            String metaHash = hashLine.substring(7);
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            Path jsonFilePath = Paths.get(String.valueOf(this.outputDir), "nvdcve-1.1-" + year + ".json");
            String hex = NistDataMirror.checksum(String.valueOf(jsonFilePath), md);
            return metaHash.equals(hex);
        }
        catch (IOException | NoSuchAlgorithmException ex) {
            ex.printStackTrace();
            return null;
        }
    }

    private static String checksum(String filepath, MessageDigest md) throws IOException {
        try (DigestInputStream dis = new DigestInputStream(new BufferedInputStream(new FileInputStream(filepath)), md);){
            int character;
            while ((character = dis.read()) != -1) {
            }
            md = dis.getMessageDigest();
        }
        byte[] digest = md.digest();
        String digestHexString = NistDataMirror.bytesToHex(digest);
        return digestHexString.toUpperCase(Locale.ROOT);
    }

    public static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    static {
        HashMap<String, String> version11Filenames = new HashMap<String, String>();
        version11Filenames.put("cveJsonModifiedUrl", CVE_JSON_11_MODIFIED_URL);
        version11Filenames.put("cveJsonRecentUrl", CVE_JSON_11_RECENT_URL);
        version11Filenames.put("cveJsonBaseUrl", CVE_JSON_11_BASE_URL);
        version11Filenames.put("cveModifiedMeta", CVE_MODIFIED_11_META);
        version11Filenames.put("cveRecentMeta", CVE_RECENT_11_META);
        version11Filenames.put("cveBaseMeta", CVE_BASE_11_META);
        versionToFilenameMaps.put("1.1", version11Filenames);
        HEX_ARRAY = "0123456789ABCDEF".toCharArray();
    }
}

