/*
 * This file is part of the OWASP Risk Rating Calculator.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package us.springett.owasp.riskrating.factors;

/**
 * This set of factors are related to the vulnerability involved. The goal here is to estimate the likelihood
 * of the particular vulnerability involved being discovered and exploited. Assume the threat agent selected above.
 *
 * @author Steve Springett
 * @since 1.0.0
 */
@SuppressWarnings("unused")
public class VulnerabilityFactor {

    private VulnerabilityFactor() {
    }

    /**
     * How easy is it for this group of threat agents to discover this vulnerability?
     */
    public enum EaseOfDiscovery implements ILikelihood {
        PRACTICALLY_IMPOSSIBLE(1),
        DIFFICULT(3),
        EASY(7),
        AUTOMATED_TOOLS_AVAILABLE(9);

        private double likelihood;

        public double getLikelihood() {
            return this.likelihood;
        }
        EaseOfDiscovery(double likelihood) {
            this.likelihood = likelihood;
        }

        public static EaseOfDiscovery fromDouble(double level) {
            for (EaseOfDiscovery easeOfDiscovery : values()) {
                if (easeOfDiscovery.likelihood == level) {
                    return easeOfDiscovery;
                }
            }
            return null;
        }

        public static EaseOfDiscovery fromString(String level) {
            return fromDouble(Double.valueOf(level));
        }
    }

    /**
     * How easy is it for this group of threat agents to actually exploit this vulnerability?
     */
    public enum EaseOfExploit implements ILikelihood {
        THEORETICAL(1),
        DIFFICULT(3),
        EASY(5),
        AUTOMATED_TOOLS_AVAILABLE(9);

        private double likelihood;

        public double getLikelihood() {
            return this.likelihood;
        }
        EaseOfExploit(double likelihood) {
            this.likelihood = likelihood;
        }

        public static EaseOfExploit fromDouble(double level) {
            for (EaseOfExploit easeOfExploit : values()) {
                if (easeOfExploit.likelihood == level) {
                    return easeOfExploit;
                }
            }
            return null;
        }

        public static EaseOfExploit fromString(String level) {
            return fromDouble(Double.valueOf(level));
        }
    }

    /**
     * How well known is this vulnerability to this group of threat agents?
     */
    public enum Awareness implements ILikelihood {
        UNKNOWN(1),
        HIDDEN(4),
        OBVIOUS(6),
        PUBLIC_KNOWLEDGE(9);

        private double likelihood;

        public double getLikelihood() {
            return this.likelihood;
        }
        Awareness(double likelihood) {
            this.likelihood = likelihood;
        }

        public static Awareness fromDouble(double level) {
            for (Awareness awareness : values()) {
                if (awareness.likelihood == level) {
                    return awareness;
                }
            }
            return null;
        }

        public static Awareness fromString(String level) {
            return fromDouble(Double.valueOf(level));
        }
    }

    /**
     * How likely is an exploit to be detected?
     */
    public enum IntrusionDetection implements ILikelihood {
        ACTIVE_DETECTION_IN_APPLICATION(1),
        LOGGED_AND_REVIEWED(3),
        LOGGED_WITHOUT_REVIEW(8),
        NOT_LOGGED(9);

        private double likelihood;

        public double getLikelihood() {
            return this.likelihood;
        }
        IntrusionDetection(double likelihood) {
            this.likelihood = likelihood;
        }

        public static IntrusionDetection fromDouble(double level) {
            for (IntrusionDetection intrusionDetection : values()) {
                if (intrusionDetection.likelihood == level) {
                    return intrusionDetection;
                }
            }
            return null;
        }

        public static IntrusionDetection fromString(String level) {
            return fromDouble(Double.valueOf(level));
        }
    }
}
