package vn.kalapa.behaviorsdk.utils

import android.os.Build
import java.security.KeyFactory
import java.security.SecureRandom
import java.security.spec.X509EncodedKeySpec
import android.util.Base64
import org.json.JSONObject
import vn.kalapa.behaviorsdk.models.KLPPair
import javax.crypto.Cipher
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec
import kotlin.collections.Map

class CryptoHelpers {

    // Decode JWT token
    @Throws(Exception::class)
    fun decodeJwtToken(jwt: String): Map<String, Any>? {
        val segments = jwt.split(".")
        if (segments.size < 2) throw Exception("Invalid JWT Token")

        val part = segments[1]
        return decodeJWTPart(part)
    }

    @Throws(Exception::class)
    private fun decodeJWTPart(value: String): Map<String, Any>? {
        val decodedBytes = base64Decode(value)
        val jsonString = String(decodedBytes, Charsets.UTF_8)
        return jsonString.toMap()
    }

    private fun base64Decode(base64: String): ByteArray {
        val base64Corrected = base64.replace("-", "+").replace("_", "/")
        val paddedBase64 = base64Corrected.padEnd((base64Corrected.length + 3) / 4 * 4, '=')
//        return Base64.getDecoder().decode(paddedBase64)
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            // For API 26 and above
            java.util.Base64.getDecoder().decode(paddedBase64)
        } else {
            // For API 21–25
            Base64.decode(paddedBase64, Base64.DEFAULT)
        }
    }

    // Encrypt Dataset
    @Throws(Exception::class)
    fun encryptDataset(token: String, dataset: Map<String, Any>): KLPPair<String>? {
        val jsonData = JSONObject(dataset).toString();
        val dataDecode = decodeJwtToken(token)
        val publicKeyBase64 = dataDecode?.get("pk") as? String ?: return null

        val randomBytes = generateRandomBytes(48)
        val aesKey = randomBytes.sliceArray(0 until 32)
        val aesIV = randomBytes.sliceArray(32 until 48)

        val encryptedData = aesEncryptMessage(jsonData, aesKey, aesIV) ?: return null

        val aesHex = randomBytes.joinToString("") { String.format("%02x", it) }
        val publicKeyData =  // Base64.getDecoder().decode(publicKeyBase64)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                // For API 26 and above
                java.util.Base64.getDecoder().decode(publicKeyBase64)
            } else {
                // For API 21–25
                Base64.decode(publicKeyBase64, Base64.DEFAULT)
            }

        val pemPublic = String(publicKeyData, Charsets.UTF_8)

        val publicKey = getPublicKeyFromPem(pemPublic)
        val encryptedAesKey = rsaEncrypt(aesHex, publicKey)
        val encryptedKeyBase64 = // Base64.getEncoder().encodeToString(encryptedAesKey)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                // For API 26 and above
                java.util.Base64.getEncoder().encodeToString(encryptedAesKey)
            } else {
                // For API 21–25
                Base64.encodeToString(encryptedAesKey, Base64.DEFAULT)
            }

        return KLPPair(encryptedKeyBase64, encryptedData)
    }

    private fun getPublicKeyFromPem(publicKeyPem: String): java.security.PublicKey {
        // Read the PEM string and parse it
        val lines = publicKeyPem
            .split("\n")
            .filter { !it.startsWith("-----BEGIN") && !it.startsWith("-----END") }
            .joinToString("")
            .replace("\\s".toRegex(), "")  // Remove any remaining whitespace

        // Decode the Base64 string to get the key bytes
        val keyBytes = // Base64.getDecoder().decode(lines)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                // For API 26 and above
                java.util.Base64.getDecoder().decode(lines)
            } else {
                // For API 21–25
                Base64.decode(lines, Base64.DEFAULT)
            }
        // Create the key specification and generate the public key
        val keySpec = X509EncodedKeySpec(keyBytes)
        val keyFactory = KeyFactory.getInstance("RSA")
        return keyFactory.generatePublic(keySpec)
    }

    @Throws(Exception::class)
    fun rsaEncrypt(data: String, publicKey: java.security.PublicKey): ByteArray {
        // Initialize RSA cipher
        val cipher = Cipher.getInstance("RSA/ECB/OAEPWithSHA1AndMGF1Padding")
        cipher.init(Cipher.ENCRYPT_MODE, publicKey)

        // Encrypt the data
        return cipher.doFinal(data.toByteArray(Charsets.UTF_8))
    }

    // Generate random bytes
    private fun generateRandomBytes(count: Int): ByteArray {
        val random = SecureRandom()
        val bytes = ByteArray(count)
        random.nextBytes(bytes)
        return bytes
    }

    // AES Encryption
    private fun aesEncryptMessage(message: String, key: ByteArray, iv: ByteArray): String? {
        return try {
            val messageBytes = message.toByteArray(Charsets.UTF_8)
            val secretKeySpec = SecretKeySpec(key, "AES")
            val ivParameterSpec = IvParameterSpec(iv)

            val cipher = Cipher.getInstance("AES/CBC/PKCS5Padding")
            cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, ivParameterSpec)

            val encryptedBytes = cipher.doFinal(messageBytes)

            encryptedBytes.joinToString("") { String.format("%02x", it) }
        } catch (e: Exception) {
            e.printStackTrace()  // Optionally log the error
            null
        }
    }

    // Pad data to block size
    private fun pad(data: ByteArray, blockSize: Int): ByteArray {
        val padding = blockSize - (data.size % blockSize)
        return data + ByteArray(padding) { padding.toByte() }
    }

    // Extension to convert Map to JSON string (mockup example)
    private fun Map<String, Any>.toJsonString(): String {
        return this.entries.joinToString(", ", "{", "}") { "\"${it.key}\":\"${it.value}\"" }
    }

    // Extension to convert JSON string to Map (mockup example)
    private fun String.toMap(): Map<String, Any> {
        return this.removeSurrounding("{", "}").split(",")
            .map { it.split(":") }
            .associate { it[0].removeSurrounding("\"") to it[1].removeSurrounding("\"") }
    }
}