package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.app.AlertDialog
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.location.Address
import android.location.Geocoder
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.os.Build
import android.os.Bundle
import android.os.Looper
import android.provider.Settings
import androidx.core.app.ActivityCompat
import com.google.android.gms.location.FusedLocationProviderClient
import com.google.android.gms.location.LocationAvailability
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import vn.kalapa.behaviorsdk.R
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.Helpers
import java.lang.NoClassDefFoundError
import java.util.Locale


class KLPGeolocationModule : IKLPModule {
    override val key: String
        get() = "location"
    val data = HashMap<String, Any>()
    private lateinit var fusedLocationClient: FusedLocationProviderClient

    private val GEOLOCATION_LATITUDE = "latitude"
    private val GEOLOCATION_LONGITUDE = "longitude"
    private val GEOLOCATION_NAME = "name"
    private val GEOLOCATION_LOCALITY = "locality"
    private val GEOLOCATION_SUB_LOCALITY = "sub_locality"
    private val GEOLOCATION_ADMINISTRATIVE_AREA = "administrative_area"
    private val GEOLOCATION_SUB_ADMINISTRATIVE_AREA = "sub_administrative_area"
    private val GEOLOCATION_POSTAL_CODE = "postal_code"
    private val GEOLOCATION_PREMISES = "premises"
    private val GEOLOCATION_ISO_COUNTRY_CODE = "country_code"
    private val GEOLOCATION_COUNTRY = "country"
    private val GEOLOCATION_FEATURE = "feature"
    private val GEOLOCATION_THOROUGHFARE = "thoroughfare"
    private val GEOLOCATION_SUB_THOROUGHFARE = "sub_thoroughfare"
    private val GEOLOCATION_WARD = "ward"
    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        return if (Common.checkIfPermissionGranted(applicationContext, Manifest.permission.ACCESS_FINE_LOCATION) && Common.checkIfPermissionGranted(applicationContext, Manifest.permission.ACCESS_COARSE_LOCATION))
            completion()
        else
            mapOf()
    }

    override fun getRequirePermission(): Array<String> {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) { // Android 10+
            arrayOf(
                Manifest.permission.ACCESS_FINE_LOCATION,
                Manifest.permission.ACCESS_COARSE_LOCATION,
                Manifest.permission.ACCESS_BACKGROUND_LOCATION
            )
        } else {
            arrayOf(
                Manifest.permission.ACCESS_FINE_LOCATION,
                Manifest.permission.ACCESS_COARSE_LOCATION
            )
        }
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        Helpers.printLog("$key KLPGeolocationModule setupCollectors")
        return getPermissionAndNeededInfo(applicationContext, completion = {
            data[key] = getGeolocationInformation(applicationContext)
            Helpers.printLog("$key KLPGeolocationModule\n ${data[key]}")
            data
        })
    }

    private fun isGPSEnabled(context: Context): Boolean {
        val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager
        return locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
    }

    private fun promptEnableGPS(context: Context) {
        val intent = Intent(Settings.ACTION_LOCATION_SOURCE_SETTINGS)
        context.startActivity(intent)
    }

    private fun getCurrentLocation(context: Context, onLocationReceived: (Location?) -> Unit) {
        try {
            val fusedLocationClient = LocationServices.getFusedLocationProviderClient(context)

            if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED
            ) {
                Helpers.printLog("getCurrentLocation onPermission not provided...")
                onLocationReceived(null)
                return
            }

            // Try to get last known location from Google Play Services
            fusedLocationClient.lastLocation.addOnSuccessListener { location: Location? ->
                if (location != null) {
                    onLocationReceived(location)
                } else {
                    // If Play Services fails, fallback to GPS & Network Provider
                    requestLocationFallback(context, onLocationReceived)
                }
            }.addOnFailureListener {
                requestLocationFallback(context, onLocationReceived)
            }
        } catch (e: NoClassDefFoundError) {
            // If Play Services is unavailable, fallback to GPS & Network
            requestLocationFallback(context, onLocationReceived)
        }
    }

    @SuppressLint("MissingPermission")
    private fun requestLocationFallback(context: Context, onLocationReceived: (Location?) -> Unit) {
        Helpers.printLog("requestLocationFallback...")
        val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager

        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
            ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED
        ) {
            Helpers.printLog("requestLocationFallback onPermission not provided...")
            onLocationReceived(null)
            return
        }

        val gpsEnabled = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
        val networkEnabled = locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)

        if (!gpsEnabled && !networkEnabled) {
            Helpers.printLog("requestLocationFallback... !gpsEnabled && !networkEnabled")
            onLocationReceived(null)
            return
        }

        val locationListener = object : LocationListener {
            override fun onLocationChanged(location: Location) {
                Helpers.printLog("requestLocationFallback... onLocationChanged ${location.longitude} ${location.latitude}")
                onLocationReceived(location)
                locationManager.removeUpdates(this) // Stop updates once location is received
            }

            override fun onStatusChanged(provider: String?, status: Int, extras: Bundle?) {}
            override fun onProviderEnabled(provider: String) {}
            override fun onProviderDisabled(provider: String) {}
        }

        // Ensure running on the main thread (Looper)
        CoroutineScope(Dispatchers.Main).launch {
            when {
                networkEnabled -> {
                    Helpers.printLog("Using Network Provider")
                    locationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 1000, 10f, locationListener)
                }

                else -> {
                    Helpers.printLog("Using GPS Provider")
                    locationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 1000, 10f, locationListener)
                }
            }
        }
    }

    private var gpsReceiver: GPSStateReceiver? = null

    private fun showGPSDialog(context: Context, onGPSChanged: (Boolean) -> Unit) {
        val builder = AlertDialog.Builder(context)
        builder.setTitle(context.getString(R.string.klp_gps_turns_off_title))
        builder.setMessage(context.getString(R.string.klp_gps_turns_off_message))

        builder.setPositiveButton(context.getString(R.string.klp_btn_confirm), null) // Set to null and override later
        builder.setNegativeButton(context.getString(R.string.klp_btn_cancel)) { dialog, _ ->
            dialog.dismiss()
            gpsReceiver?.let { unregisterGPSReceiver(context, it) }
            onGPSChanged(false)
        }

        val dialog = builder.create()
        dialog.show()

        // Override positive button behavior to prevent auto-dismiss
        dialog.getButton(AlertDialog.BUTTON_POSITIVE).setOnClickListener {
            gpsReceiver = registerGPSReceiver(context) { isGPSOn ->
                if (isGPSOn) {
                    Helpers.printLog("GPS Check", "GPS is now enabled!")
                    gpsReceiver?.let { unregisterGPSReceiver(context, it) }
                    dialog.dismiss() // Now dismiss the dialog only when GPS is enabled
                    onGPSChanged(true)
                } else {
                    Helpers.printLog("GPS Check", "GPS is still disabled.")
                    onGPSChanged(false)
                }
            }
            promptEnableGPS(context) // Open GPS settings
        }
    }

    /**
     * Requests a fresh location update if last known location is not available.
     */
    private fun requestFreshLocation(
        applicationContext: Context,
        fusedLocationClient: FusedLocationProviderClient,
        onLocationReceived: (Location?) -> Unit
    ) {
        val locationRequest: LocationRequest = if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.S) {
            LocationRequest.Builder(Priority.PRIORITY_HIGH_ACCURACY, 60000).apply {
                setMinUpdateIntervalMillis(5000) // Fastest interval
                setMaxUpdates(1) // Ensure only one update
            }.build()
        } else {
            LocationRequest.create().apply {
                interval = 60000
                fastestInterval = 5000
                priority = LocationRequest.PRIORITY_HIGH_ACCURACY
                numUpdates = 1
            }
        }


        val locationCallback = object : LocationCallback() {
            override fun onLocationResult(locationResult: LocationResult) {
                val location = locationResult.lastLocation
                Helpers.printLog("$key Received fresh location: $location")
                onLocationReceived(location)
                fusedLocationClient.removeLocationUpdates(this) // Stop updates after first location
            }

            override fun onLocationAvailability(locationAvailability: LocationAvailability) {
                if (!locationAvailability.isLocationAvailable) {
                    Helpers.printLog("$key Location not available.")
                    onLocationReceived(null)
                    fusedLocationClient.removeLocationUpdates(this)
                }
            }
        }

        if (ActivityCompat.checkSelfPermission(applicationContext, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            onLocationReceived(null)
            return
        }

        // Request a fresh location update
        fusedLocationClient.requestLocationUpdates(locationRequest, locationCallback, Looper.getMainLooper())
    }


    private fun getGeolocationInformation(applicationContext: Context): Map<String, Any> {
        var longitude: Double = -1.0
        var latitude: Double = -1.0
        var address: Address? = null
        var doneGetAddress = false

        getCurrentLocation(applicationContext) { location ->
            if (location != null) {
                longitude = location.longitude
                latitude = location.latitude
                address = getAddressFromCoordinates(applicationContext, latitude, longitude) {
                    doneGetAddress = true
                }
            } else {
                longitude = 0.0
                latitude = 0.0
            }
        }
        while (true) {
            if (longitude == 0.0 || doneGetAddress)
                break
            else Thread.sleep(100)
        }
        Helpers.printLog("$key getGeolocationInformation done getCurrentLocation $longitude $latitude $address")
        return mapOf(
            GEOLOCATION_LATITUDE to latitude,
            GEOLOCATION_LONGITUDE to longitude,
            GEOLOCATION_NAME to (address?.getAddressLine(0) ?: ""),
            GEOLOCATION_LOCALITY to (address?.locality ?: ""),
            GEOLOCATION_SUB_LOCALITY to (address?.subLocality ?: ""),
            GEOLOCATION_ADMINISTRATIVE_AREA to (address?.adminArea ?: ""),
            GEOLOCATION_SUB_ADMINISTRATIVE_AREA to (address?.subAdminArea ?: ""),
            GEOLOCATION_PREMISES to (address?.premises ?: ""),
            GEOLOCATION_POSTAL_CODE to (address?.postalCode ?: ""),
            GEOLOCATION_ISO_COUNTRY_CODE to (address?.countryCode ?: ""),
            GEOLOCATION_COUNTRY to (address?.countryName ?: ""),
            GEOLOCATION_FEATURE to (address?.featureName ?: ""),
            GEOLOCATION_THOROUGHFARE to (address?.thoroughfare ?: ""),
            GEOLOCATION_SUB_THOROUGHFARE to (address?.subThoroughfare ?: ""),
            GEOLOCATION_WARD to getOrFillWardIfMissing(address)
        )
    }

    private fun getAddressFromCoordinates(context: Context, latitude: Double, longitude: Double, completion: () -> Unit): Address? {
        val geocoder = Geocoder(context, Locale.getDefault())
        try {
            val addresses = geocoder.getFromLocation(latitude, longitude, 1)
            Thread.sleep(2000)
            if (!addresses.isNullOrEmpty()) {
//                Helpers.printLog("$key getAddressFromCoordinates ${addresses[0]}")
                completion()
                return addresses[0]
//                return addresses[0].getAddressLine(0) // Full address
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        completion()
        return null
    }

    private fun getOrFillWardIfMissing(address: Address?): String {
        return if (address == null)
            ""
        else if (!address.locality.isNullOrEmpty())
            address.locality
        else if (!address.subLocality.isNullOrEmpty())
            address.subLocality
        else {
            val addressName = address.getAddressLine(0)  // Full address: "58 P. Tố Hữu, Trung Văn, Nam Từ Liêm, Hà Nội, Vietnam"
            val parts = addressName.split(",").map { it.trim() }
            return if (parts.size > 2) parts[1] else ""  // Extract "Trung Văn"
        }
    }

}

class GPSStateReceiver(private val onGPSChanged: (Boolean) -> Unit) : BroadcastReceiver() {
    override fun onReceive(context: Context?, intent: Intent?) {
        if (intent?.action == LocationManager.PROVIDERS_CHANGED_ACTION) {
            val locationManager = context?.getSystemService(Context.LOCATION_SERVICE) as LocationManager
            val isGPSOn = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
            Helpers.printLog("GPSStateReceiver", "GPS state changed: $isGPSOn")
            onGPSChanged(isGPSOn) // Notify the listener
        }
    }
}

fun registerGPSReceiver(context: Context, onGPSChanged: (Boolean) -> Unit): GPSStateReceiver {
    val receiver = GPSStateReceiver(onGPSChanged)
    val intentFilter = IntentFilter(LocationManager.PROVIDERS_CHANGED_ACTION)
    context.registerReceiver(receiver, intentFilter)
    return receiver
}

// Unregister when no longer needed
fun unregisterGPSReceiver(context: Context, receiver: GPSStateReceiver) {
    context.unregisterReceiver(receiver)
}