package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.app.Activity
import android.content.Context
import android.database.Cursor
import android.net.Uri
import android.os.Build
import android.provider.MediaStore
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.Helpers

class KLPMusicModule : IKLPModule {
    override val key: String
        get() = "music"

    override fun getRequirePermission(): Array<String> {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            arrayOf(Manifest.permission.READ_MEDIA_AUDIO)
        } else {
            arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE)
        }
    }


    private val MUSIC_TOTAL_ARTIST_COUNT = "total_artists_count"
    private val MUSIC_TOTAL_PLAY_COUNT = "total_play_count"
    private val MUSIC_UNIQUE_GENRES_COUNT = "unique_genres_count"
    private val MUSIC_AVERAGE_PLAY_COUNT = "average_play_count"
    private val MUSIC_TOTAL_ALBUM_COUNT = "total_albums_count"
    private val MUSIC_AUDIO_FILES_COUNT = "audio_files_count"
    private val MUSIC_FILES_COUNT = "music_files_count"

    val data = HashMap<String, Any>()

    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        return if (Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_MEDIA_AUDIO) ||
            Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_EXTERNAL_STORAGE)
        ) {
            completion()
        } else {
            mapOf()
        }

    }


    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        return getPermissionAndNeededInfo(applicationContext, completion = {
            data[key] = getMusicInfo(applicationContext)
            data
        })
    }

    private fun getMusicInfo(applicationContext: Context): Map<String, Any> {
        return mapOf(
            MUSIC_TOTAL_ARTIST_COUNT to getTotalArtistsCount(applicationContext),
            MUSIC_AVERAGE_PLAY_COUNT to -1,
            MUSIC_TOTAL_PLAY_COUNT to -1,
            MUSIC_UNIQUE_GENRES_COUNT to getTotalGenresCount(applicationContext),
            MUSIC_TOTAL_ALBUM_COUNT to getTotalAlbumsCount(applicationContext),
            MUSIC_AUDIO_FILES_COUNT to getAudioFilesCount(applicationContext),
            MUSIC_FILES_COUNT to getMusicFilesCount(applicationContext)
        )
    }

    private fun getTotalArtistsCount(applicationContext: Context): Int {
        val uri: Uri = MediaStore.Audio.Artists.EXTERNAL_CONTENT_URI
        val projection = arrayOf(MediaStore.Audio.Artists._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, null, null, null)
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getTotalAlbumsCount(applicationContext: Context): Int {
        val uri: Uri = MediaStore.Audio.Albums.EXTERNAL_CONTENT_URI
        val projection = arrayOf(MediaStore.Audio.Albums._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, null, null, null)
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getTotalGenresCount(applicationContext: Context): Int {
        val uri: Uri = MediaStore.Audio.Genres.EXTERNAL_CONTENT_URI
        val projection = arrayOf(MediaStore.Audio.Genres._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, null, null, null)
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getAudioFilesCount(applicationContext: Context): Int {
        val uri: Uri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI
        val projection = arrayOf(MediaStore.Audio.Media._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, null, null, null)
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getMusicFilesCount(applicationContext: Context): Int {
        val uri: Uri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI
        val selection = MediaStore.Audio.Media.IS_MUSIC + " != 0"
        val projection = arrayOf(MediaStore.Audio.Media._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, selection, null, null)
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private var totalPlayCount: Int = 0
    private fun getAveragePlayCount(applicationContext: Context): Float {
        val uri: Uri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI
        val projection = arrayOf(MediaStore.Audio.Media._ID, MediaStore.Audio.Media._COUNT)
        val cursor: Cursor? = applicationContext.contentResolver.query(uri, projection, null, null, null)
        var totalSongs = 0

        cursor?.use {
            while (it.moveToNext()) {
                val playCount = it.getInt(it.getColumnIndexOrThrow(MediaStore.Audio.Media._COUNT))
                totalPlayCount += playCount
                totalSongs++
            }
        }

        return if (totalSongs > 0) totalPlayCount.toFloat() / totalSongs else 0f
    }

}