package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.app.Activity
import android.content.ContentUris
import android.content.Context
import android.content.pm.PackageManager
import android.database.Cursor
import android.net.Uri
import android.provider.CalendarContract
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import vn.kalapa.behaviorsdk.utils.Common
import java.util.Calendar


class KLPReminderModule : IKLPModule {
    override val key: String
        get() = "reminder"
    val data = HashMap<String, Any>()

    private val REMINDER_COUNT = "reminder_count"
    private val REMINDER_FOR_30_DAYS_COUNT = "reminders_for_30_days_count"
    private val REMINDER_WORKTIME_COUNT = "work_time_reminder_count"
    private val REMINDER_REPETITIVE_COUNT = "repetitive_reminder_count"
    private val REMINDER_WEEKDAY_COUNT = "weekday_reminder_count"
    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        return if (Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_CALENDAR))
            completion()
        else
            mapOf()
    }

    override fun getRequirePermission(): Array<String> {
        return arrayOf(
            Manifest.permission.READ_PHONE_STATE,
            Manifest.permission.READ_CALENDAR
        )
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        return getPermissionAndNeededInfo(applicationContext, completion = {
            data[key] = getReminderInformation(applicationContext)
            data
        })
    }

    private fun getReminderInformation(applicationContext: Context): Map<String, Int> {
        return mapOf(
            REMINDER_COUNT to getReminderCount(applicationContext),
            REMINDER_WEEKDAY_COUNT to getWeekdayReminderCount(applicationContext),
            REMINDER_WORKTIME_COUNT to getWorkTimeReminderCount(applicationContext),
            REMINDER_FOR_30_DAYS_COUNT to getRemindersForLast30DaysCount(applicationContext),
            REMINDER_REPETITIVE_COUNT to getRepetitiveReminderCount(applicationContext)
        )
    }

    private fun getReminderCount(applicationContext: Context): Int {
        val builder = Uri.parse("content://com.android.calendar/instances/when").buildUpon()
        ContentUris.appendId(builder, Long.MIN_VALUE)
        ContentUris.appendId(builder, Long.MAX_VALUE)
        //Uri.parse("content://com.android.calendar/calendars")
        val cursor: Cursor? = applicationContext.contentResolver.query(
            builder.build(),
            (arrayOf("_id")), null, null, null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getRemindersForLast30DaysCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Reminders._ID)
        val now = System.currentTimeMillis()
        val in30Days = now - 30 * 24 * 60 * 60 * 1000L // 30 days in milliseconds

        val selection = "${CalendarContract.Instances.BEGIN} >= ? AND ${CalendarContract.Instances.BEGIN} <= ?"
        val selectionArgs = arrayOf(in30Days.toString(), now.toString())
        val builder = Uri.parse("content://com.android.calendar/instances/when").buildUpon()
        ContentUris.appendId(builder, Long.MIN_VALUE)
        ContentUris.appendId(builder, Long.MAX_VALUE)
        //Uri.parse("content://com.android.calendar/calendars")
        val cursor: Cursor? = applicationContext.contentResolver.query(
            builder.build(),
            projection, selection, selectionArgs, null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getWorkTimeReminderCount(applicationContext: Context): Int {
        val projection = arrayOf(
            CalendarContract.Reminders._ID,
            CalendarContract.Instances.BEGIN
        )

        val selection = "${CalendarContract.Instances.BEGIN} >= ?"
        val selectionArgs = arrayOf(System.currentTimeMillis().toString())

        val builder = Uri.parse("content://com.android.calendar/instances/when").buildUpon()
        ContentUris.appendId(builder, Long.MIN_VALUE)
        ContentUris.appendId(builder, Long.MAX_VALUE)
        //Uri.parse("content://com.android.calendar/calendars")
        val cursor: Cursor? = applicationContext.contentResolver.query(
            builder.build(),
            projection, selection, selectionArgs, null
        )
        var count = 0
        cursor?.use {
            while (it.moveToNext()) {
                val beginTime = it.getLong(it.getColumnIndexOrThrow(CalendarContract.Instances.BEGIN))
                val calendar = Calendar.getInstance().apply { timeInMillis = beginTime }
                val hour = calendar.get(Calendar.HOUR_OF_DAY)
                if (hour in 9..17) { // 9 AM to 5 PM
                    count++
                }
            }
        }
        return count
    }

    private fun getRepetitiveReminderCount(applicationContext: Context): Int {
        val projection = arrayOf(
            CalendarContract.Reminders._ID,
            CalendarContract.Events.RRULE
        )

        val selection = "${CalendarContract.Events.RRULE} IS NOT NULL"
        val builder = Uri.parse("content://com.android.calendar/instances/when").buildUpon()
        ContentUris.appendId(builder, Long.MIN_VALUE)
        ContentUris.appendId(builder, Long.MAX_VALUE)
        //Uri.parse("content://com.android.calendar/calendars")
        val cursor: Cursor? = applicationContext.contentResolver.query(
            builder.build(),
            projection, selection, null, null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getWeekdayReminderCount(applicationContext: Context): Int {
        val projection = arrayOf(
            CalendarContract.Reminders._ID,
            CalendarContract.Instances.BEGIN
        )

        val selection = "${CalendarContract.Instances.BEGIN} >= ?"
        val selectionArgs = arrayOf(System.currentTimeMillis().toString())
        val builder = Uri.parse("content://com.android.calendar/instances/when").buildUpon()
        ContentUris.appendId(builder, Long.MIN_VALUE)
        ContentUris.appendId(builder, Long.MAX_VALUE)
        //Uri.parse("content://com.android.calendar/calendars")
        val cursor: Cursor? = applicationContext.contentResolver.query(
            builder.build(),
            projection, selection, selectionArgs, null
        )

        var count = 0
        cursor?.use {
            while (it.moveToNext()) {
                val beginTime = it.getLong(it.getColumnIndexOrThrow(CalendarContract.Instances.BEGIN))
                val calendar = Calendar.getInstance().apply { timeInMillis = beginTime }
                val dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK)
                if (dayOfWeek in Calendar.MONDAY..Calendar.FRIDAY) {
                    count++
                }
            }
        }
        return count
    }
}
