package vn.kalapa.behaviorsdk.models

import org.json.JSONArray
import org.json.JSONObject

open class KLPBehaviorObject private constructor(
    val event: String,
    val time: Long,
    val target: String?
) {
    open fun toJson(): JSONObject {
        return JSONObject().apply {
            put("event", event)
            put("time", time)
            put("target", target)
        }
    }

    override fun toString(): String {
        return "KLPBehaviorObject(event='$event', time=$time, target=$target)"
    }

    companion object {
        fun fromJson(json: JSONObject): KLPBehaviorObject {
            val event = json.getString("event")
            val time = json.getLong("time")
            val target = json.optString("target", null)

            return when (event) {
                KLPBehaviorObjectEnum.POINTER.event -> KLPBehaviorPointer.fromJson(json)
                KLPBehaviorObjectEnum.KEYBOARD.event -> KLPBehaviorKeyboard.fromJson(json)
                KLPBehaviorObjectEnum.CLICK.event -> KLPBehaviorClick.fromJson(json)
                else -> KLPBehaviorObject(event, time, target)
            }
        }
    }

    class KLPBehaviorPointer(
        target: String,
        val duration: Long,
        val distance: Int,
        val position: Array<KLPBehaviorPosition>
    ) : KLPBehaviorObject(KLPBehaviorObjectEnum.POINTER.event, System.currentTimeMillis(), target) {
        override fun toString(): String {
            return "KLPBehaviorPointer(event='$event', time=$time, target=$target, duration=$duration, distance=$distance, position=${position.contentToString()})"
        }
        override fun toJson(): JSONObject {
            return super.toJson().apply {
                put("duration", duration)
                put("distance", distance)
                put("position", JSONArray(position.map { it.toJson() }))
            }
        }

        companion object {
            fun fromJson(json: JSONObject): KLPBehaviorPointer {
                val target = json.getString("target")
                val duration = json.getLong("duration")
                val distance = json.getInt("distance")
                val positionArray = json.getJSONArray("position")
                val position = Array(positionArray.length()) {
                    KLPBehaviorPosition.fromJson(positionArray.getJSONObject(it))
                }
                return KLPBehaviorPointer(target, duration, distance, position)
            }
        }
    }

    open class KLPBehaviorKeyboard(
        target: String,
        val key: String,
        val position: KLPBehaviorPosition
    ) : KLPBehaviorObject(KLPBehaviorObjectEnum.KEYBOARD.event, System.currentTimeMillis(), target) {
        override fun toString(): String {
            return "KLPBehaviorKeyboard(event='$event', time=$time, target=$target, key='$key', position=$position)"
        }
        override fun toJson(): JSONObject {
            return super.toJson().apply {
                put("key", key)
                put("position", position.toJson())
            }
        }

        companion object {
            fun fromJson(json: JSONObject): KLPBehaviorKeyboard {
                val target = json.getString("target")
                val key = json.getString("key")
                val position = KLPBehaviorPosition.fromJson(json.getJSONObject("position"))
                return KLPBehaviorKeyboard(target, key, position)
            }
        }
    }

    class KLPBehaviorAutofill(
        target: String,
        key: String,
        position: KLPBehaviorPosition,
        val type: String = "autofill"
    ) : KLPBehaviorKeyboard(target, key, position) {
        override fun toString(): String {
            return "KLPBehaviorAutofill(event='$event', time=$time, target=$target, key='$key', position=$position, type='$type')"
        }
        override fun toJson(): JSONObject {
            return super.toJson().apply {
                put("type", type)
            }
        }

        companion object {
            fun fromJson(json: JSONObject): KLPBehaviorAutofill {
                val target = json.getString("target")
                val key = json.getString("key")
                val position = KLPBehaviorPosition.fromJson(json.getJSONObject("position"))
                val type = json.optString("type", "autofill")
                return KLPBehaviorAutofill(target, key, position, type)
            }
        }
    }

    class KLPBehaviorClick(
        target: String,
        val position: KLPBehaviorPosition
    ) : KLPBehaviorObject(KLPBehaviorObjectEnum.CLICK.event, System.currentTimeMillis(), target) {
        override fun toString(): String {
            return "KLPBehaviorClick(event='$event', time=$time, target=$target, position=$position)"
        }
        override fun toJson(): JSONObject {
            return super.toJson().apply {
                put("position", position.toJson())
            }
        }

        companion object {
            fun fromJson(json: JSONObject): KLPBehaviorClick {
                val target = json.getString("target")
                val position = KLPBehaviorPosition.fromJson(json.getJSONObject("position"))
                return KLPBehaviorClick(target, position)
            }
        }
    }

    class KLPBehaviorPosition(val x: Int, val y: Int) {
        fun toJson(): JSONObject {
            return JSONObject().apply {
                put("x", x)
                put("y", y)
            }
        }
        override fun toString(): String {
            return "KLPBehaviorPosition(x=$x, y=$y)"
        }
        companion object {
            fun fromJson(json: JSONObject): KLPBehaviorPosition {
                val x = json.getInt("x")
                val y = json.getInt("y")
                return KLPBehaviorPosition(x, y)
            }
        }
    }

    private enum class KLPBehaviorObjectEnum(val event: String) {
        POINTER("pointer"),
        KEYBOARD("keyboard"),
        CLICK("click")
    }
}
