package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.app.Activity
import android.content.ContentResolver
import android.content.Context
import android.content.pm.PackageManager
import android.database.Cursor
import android.provider.CalendarContract
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.Helpers
import java.util.Calendar

class KLPCalendarModule : IKLPModule {
    override val key: String
        get() = "calendar"
    val data = HashMap<String, Any>()

    private val CALENDAR_EVENT_COUNT = "event_count"
    private val CALENDAR_EVENT_FOR_30_DAYS = "events_for_30_days_count"
    private val CALENDAR_ORGANIZED_EVENT_COUNT = "organized_event_count"
    private val CALENDAR_REPETITIVE_EVENTS_FOR_30_DAYS_COUNT = "repetitive_events_for_30_days_count"
    private val CALENDAR_WEEKDAY_EVENT_COUNT = "weekday_event_count"
    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        return if (Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_CALENDAR)) completion()
        else mapOf()
    }

    override fun getRequirePermission(): Array<String> {
        return arrayOf(Manifest.permission.READ_CALENDAR,Manifest.permission.READ_PHONE_STATE)
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        return getPermissionAndNeededInfo(applicationContext, completion = {
            data[key] = getCalendarInformation(applicationContext)
            data
        })
    }

    private fun getCalendarInformation(applicationContext: Context): Map<String, Int> {
        return mapOf(
            CALENDAR_EVENT_COUNT to getEventCount(applicationContext),
            CALENDAR_EVENT_FOR_30_DAYS to getEventsForLast30DaysCount(applicationContext),
            CALENDAR_ORGANIZED_EVENT_COUNT to getOrganizedEventCount(applicationContext),
            CALENDAR_REPETITIVE_EVENTS_FOR_30_DAYS_COUNT to getRepetitiveEventsForLast30DaysCount(applicationContext),
            CALENDAR_WEEKDAY_EVENT_COUNT to getWeekdayEventCount(applicationContext)
        )

    }

    private fun getEventCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Events._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            null,
            null,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getEventsForLast30DaysCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Events._ID)
        val now = System.currentTimeMillis()
        val in30Days = now - 30 * 24 * 60 * 60 * 1000L // 30 days in milliseconds
//        Helpers.printLog("getEventsForLast30DaysCount $now $in30Days")
        val selection = "${CalendarContract.Events.DTSTART} >= ? AND ${CalendarContract.Events.DTSTART} <= ?"
        val selectionArgs = arrayOf(in30Days.toString(), now.toString())

        val cursor: Cursor? = applicationContext.contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getOrganizedEventCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Events._ID)
        val selection = "${CalendarContract.Events.ORGANIZER} = ?"
        val selectionArgs = arrayOf("your.email@example.com") // Replace with the user's email

        val cursor: Cursor? = applicationContext.contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getRepetitiveEventsForLast30DaysCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Events._ID)
        val now = System.currentTimeMillis()
        val in30Days = now - 30 * 24 * 60 * 60 * 1000L
//        Helpers.printLog("getEventsForLast30DaysCount $now $in30Days")
        val selection = "${CalendarContract.Events.DTSTART} >= ? AND ${CalendarContract.Events.DTSTART} <= ? AND ${CalendarContract.Events.RRULE} IS NOT NULL"
        val selectionArgs = arrayOf(in30Days.toString(), now.toString())

        val cursor: Cursor? = applicationContext.contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getWeekdayEventCount(applicationContext: Context): Int {
        val projection = arrayOf(CalendarContract.Events._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            null,
            null,
            null
        )

        var count = 0
        cursor?.use {
            while (it.moveToNext()) {
                try {
                    val start = it.getLong(it.getColumnIndexOrThrow(CalendarContract.Events.DTSTART))
                    val calendar = Calendar.getInstance()
                    calendar.timeInMillis = start
                    val dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK)
                    if (dayOfWeek in Calendar.MONDAY..Calendar.FRIDAY) {
                        count++
                    }
                } catch (_: IllegalArgumentException) {
                }
            }
        }
        return count
    }
}