package vn.kalapa.behaviorsdk.managers

import android.accounts.Account
import android.accounts.AccountManager
import android.content.Context
import android.os.Build
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.google.android.gms.common.ConnectionResult
import com.google.android.gms.common.GoogleApiAvailability
import com.google.android.gms.common.GooglePlayServicesNotAvailableException
import com.google.android.gms.common.GooglePlayServicesRepairableException
import vn.kalapa.behaviorsdk.utils.Helpers
import java.io.IOException
import java.lang.NoClassDefFoundError
import java.lang.SecurityException
import java.security.MessageDigest


object UUIDManager {
    fun getOrSetUUID(context: Context): String {
        val uuid = getGADeviceId(context) ?: getDeviceId()
//        UUIDAccountManager.setUUID(context, uuid) // ✅ Store both original and backup UUID
        return uuid
    }

    fun getGADeviceId(context: Context): String? {
        try {
            if (GooglePlayServicesHelper.isGooglePlayServicesAvailable(context)) {
                try {
                    val aaid = GooglePlayServicesHelper.getAAID(context)
                    if (aaid != null) {
                        Helpers.printLog("GooglePlay Service is Okay. aaid: $aaid")
                        return aaid
                    } else Helpers.printLog("GooglePlay Service is Okay but AAID return null")
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            } else
                Helpers.printLog("GooglePlay Service is missing or not available")
        } catch (e: NoClassDefFoundError) {
            Helpers.printLog("GooglePlay Service is not implemented")
        }
        return null
    }

    fun getDeviceId(): String {
        val rawId = buildString {
            append(Build.MANUFACTURER)  // e.g., "Google"
            append("-")
            append(Build.MODEL)  // e.g., "Pixel 6"
            append("-")
            append(Build.DEVICE)  // e.g., "redfin"
            append("-")
            append(Build.PRODUCT)  // e.g., "redfin"
            append("-")
            append(Build.VERSION.SDK_INT)  // e.g., "30"
            append("-")
            append(Build.BOARD)  // e.g., "TRINKET"
            append("-")
            append(Build.HARDWARE)  // e.g., "qcom"
            append("-")
            append(Build.FINGERPRINT)  // Contains unique OS build info
        }

        return hashString(rawId) // Hash the string for security and consistency
    }

    private fun hashString(input: String): String {
        val bytes = MessageDigest.getInstance("SHA-256").digest(input.toByteArray())
        return bytes.joinToString("") { "%02x".format(it) } // Convert to hex string
    }


}

object GooglePlayServicesHelper {
    fun isGooglePlayServicesAvailable(context: Context): Boolean {
        try {
            val googleApiAvailability = GoogleApiAvailability.getInstance()
            val resultCode = googleApiAvailability.isGooglePlayServicesAvailable(context)
            return resultCode == ConnectionResult.SUCCESS
        } catch (e: NoClassDefFoundError) {
            return false
        }
    }

    fun getAAID(context: Context): String? {
        return try {
            val adInfo: AdvertisingIdClient.Info = AdvertisingIdClient.getAdvertisingIdInfo(context)
            adInfo.id // Returns the Advertising ID
        } catch (e: GooglePlayServicesNotAvailableException) {
            // Google Play Services is not available on the device.
            // Handle this case appropriately, e.g., log the error or return null.
            println("Google Play Services not available: ${e.message}")
            null
        } catch (e: GooglePlayServicesRepairableException) {
            // Google Play Services needs to be updated.
            // Handle this case appropriately, e.g., prompt the user to update.
            println("Google Play Services needs repair: ${e.message}")
            null
        } catch (e: IOException) {
            // Network or I/O error occurred.
            // Handle this case appropriately, e.g., retry or return null.
            println("IO error occurred: ${e.message}")
            null
        } catch (e: IllegalStateException) {
            // The call is not made on the main thread.
            println("Illegal State Exception: ${e.message}")
            null
        } catch (e: NoClassDefFoundError) {
            // The call is not made on the main thread.
            println("NoClassDefFoundError Exception: ${e.message}")
            null
        } catch (e: Exception) {
            // General exception handling.
            println("An unexpected error occurred: ${e.message}")
            null
        }
    }
}

object UUIDAccountManager {
    private const val ACCOUNT_TYPE = "vn.kalapa.sdk.uuid"
    private const val ACCOUNT_NAME = "SDK_UUID_ACCOUNT"
    private const val UUID_KEY = "SDK_DEVICE_UUID"

    fun getUUID(context: Context): String? {
        val accountManager = AccountManager.get(context)
        val accounts = accountManager.getAccountsByType(ACCOUNT_TYPE)
        return accounts.firstOrNull()?.let {
            accountManager.getUserData(it, UUID_KEY)
        }
    }

    fun setUUID(context: Context, uuid: String) {
        try {
            val accountManager = AccountManager.get(context)
            val accounts = accountManager.getAccountsByType(ACCOUNT_TYPE)

            if (accounts.isEmpty()) {
                val account = Account(ACCOUNT_NAME, ACCOUNT_TYPE)
                accountManager.addAccountExplicitly(account, null, null)
            }

            accounts.forEach { account ->
                accountManager.setUserData(account, UUID_KEY, uuid)
            }
        } catch (exception: SecurityException) {
            exception.printStackTrace()
            Helpers.printLog("setUUID ${exception.message}")
        }
    }


}

