package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.content.Context
import android.location.LocationManager
import android.os.Build
import kotlinx.coroutines.runBlocking
import vn.kalapa.behaviorsdk.models.KLPPair
import vn.kalapa.behaviorsdk.models.KLPTelephonyInfo
import vn.kalapa.behaviorsdk.service.KLPBehaviorScore
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.Common.Companion.formatSizeInGB
import vn.kalapa.behaviorsdk.utils.Helpers
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.DEVICE_ANDROID
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NA
import java.util.Locale
import java.util.TimeZone
import kotlin.math.sqrt

// done
class KLPCoreModule : IKLPModule {
    private val COLLECTOR_DEVICE = "device"
    private val COLLECTOR_STATUS = "status"
    private val COLLECTOR_NETWORK = "network"
    private val COLLECTOR_BATTERY = "battery"
    private val COLLECTOR_LOCALE = "locale"
    private val COLLECTOR_SCREEN = "screen"
    private val COLLECTOR_CPU = "cpu"
    private val COLLECTOR_DISK = "disk"
    private val COLLECTOR_MEMORY = "memory"
    private var data = HashMap<String, Any>()
    override val key: String get() = "core"
    private var telephonyInfo: KLPTelephonyInfo? = null

    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        telephonyInfo = Common.getTelephonyInfo(applicationContext)
        return completion()
    }

    override fun getRequirePermission(): Array<String> {
        return arrayOf(
            Manifest.permission.READ_PHONE_STATE
        )
//        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {  // Android 12+
//            Manifest.permission.READ_PHONE_STATE,
//            Manifest.permission.BLUETOOTH_CONNECT,
//            Manifest.permission.BLUETOOTH_SCAN
//            )
//        } else {
//            arrayOf(
//                Manifest.permission.READ_PHONE_STATE,
//                Manifest.permission.BLUETOOTH,
//                Manifest.permission.BLUETOOTH_ADMIN
//            )
//        }
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        return getPermissionAndNeededInfo(applicationContext, completion = {
            Helpers.printLog("KLPCoreComponent setupCollectors...")
            data[COLLECTOR_STATUS] = collectPhoneStatus(applicationContext) // done
            data[COLLECTOR_DEVICE] = collectDeviceInfo() // done
            data[COLLECTOR_MEMORY] = collectMemoryInfo(applicationContext) // done
            data[COLLECTOR_DISK] = collectDiskInfo() // done
            data[COLLECTOR_BATTERY] = collectBatteryInfo(applicationContext) // done
            data[COLLECTOR_LOCALE] = collectLocaleInfo() // done
            data[COLLECTOR_SCREEN] = collectScreenInfo(applicationContext) // done
            data[COLLECTOR_NETWORK] = collectNetworkInfo() // done
            data[COLLECTOR_CPU] = collectCPUInfo() // done
//            Helpers.printLog("COLLECTOR_STATUS ${data[COLLECTOR_STATUS].toString()}")
//            Helpers.printLog("COLLECTOR_DEVICE ${data[COLLECTOR_DEVICE].toString()}")
//            Helpers.printLog("COLLECTOR_NETWORK ${data[COLLECTOR_NETWORK].toString()}")
//            Helpers.printLog("COLLECTOR_BATTERY ${data[COLLECTOR_BATTERY].toString()}")
//            Helpers.printLog("COLLECTOR_LOCALE ${data[COLLECTOR_LOCALE].toString()}")
//            Helpers.printLog("COLLECTOR_SCREEN ${data[COLLECTOR_SCREEN]}")
//            Helpers.printLog("COLLECTOR_CPU ${data[COLLECTOR_CPU].toString()}")
//            Helpers.printLog("COLLECTOR_DISK ${data[COLLECTOR_DISK].toString()}")
//            Helpers.printLog("COLLECTOR_MEMORY ${data[COLLECTOR_MEMORY].toString()}")
            val collectorResult = mapOf(key to data)
            Helpers.printLog("KLPCoreComponent done collected... $collectorResult")
            collectorResult
        })
    }

    private fun collectPhoneStatus(context: Context): Map<String, Any> {
        val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager
        val IS_EMULATOR_DEVICE = "is_emulator_device"
        val IS_LOCATION_ENABLED = "is_location_enabled"
        val IS_GPS_ENABLED = "is_gps_enabled"
        val IS_ACCESSIBILITY_ENABLED = "is_accessibility_enabled"
        val NETWORK_CONNECTION_TYPE = "network_connection_type"
        val IS_WIFI_ENABLED = "is_wifi_enabled"
        val IS_ROAMING_ENABLED = "is_roaming_enabled"
        val IS_BETTERY_CHARGING = "is_battery_charging"
        val IS_ROOTED_DEVICE = "is_rooted_device" // new since 1.0.6.4
        val FP_FINGERPRINT = "fp_fingerprint" // new since 1.0.6.4
        val UNIQUE_DEVICE_ID = "unique_device_id"
        val GA_DEVICE_ID = "ga_device_id"
        val AG_DEVICE_ID = "ag_device_id"
        val S_DEVICE_ID = "s_device_id"
        val SDK_VERSION = "sdk_version"
        var data: Map<String, Any>
        runBlocking { // Ensure coroutine executes synchronously
            val fp = Common.getFingerprint(context)
            data = mapOf(
                FP_FINGERPRINT to fp,
                UNIQUE_DEVICE_ID to (telephonyInfo?.uniqueDeviceId ?: NA),
                GA_DEVICE_ID to (telephonyInfo?.gaDeviceId ?: NA),
                AG_DEVICE_ID to (telephonyInfo?.agDeviceId ?: NA),
                S_DEVICE_ID to (telephonyInfo?.sDeviceId ?: NA),
                IS_EMULATOR_DEVICE to (Common.isRunningOnEmulator() ?: false),
                IS_ROOTED_DEVICE to Common.isRootedByCheckRootPackages(context),
                SDK_VERSION to KLPBehaviorScore.getSDKVersion(),
                IS_GPS_ENABLED to locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER),
                IS_LOCATION_ENABLED to if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) locationManager.isLocationEnabled else NA,
                IS_WIFI_ENABLED to locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER),
                IS_EMULATOR_DEVICE to (Common.isRunningOnEmulator() ?: NA),
                NETWORK_CONNECTION_TYPE to Common.getNetworkConnectionType(context),
                IS_ROAMING_ENABLED to (telephonyInfo?.isRoamingEnabled ?: NA),
                IS_BETTERY_CHARGING to Common.isBatteryCharging(context),
                IS_ACCESSIBILITY_ENABLED to false //Common.isAccessibilityEnabled(context)
            )
        }
        return data
    }

    private fun collectMemoryInfo(context: Context): Map<String, Any> {
        val PHYSICAL_MEMORY = "physical_memory"
        val AVAILABLE_MEMORY = "available_memory"
        val LOW_MEMORY = "low_memory"
        val memoryInfo = Common.getMemoryInfo(context)
        return mapOf(
            PHYSICAL_MEMORY to formatSizeInGB(memoryInfo.totalMem),
            AVAILABLE_MEMORY to formatSizeInGB(memoryInfo.availMem),
            LOW_MEMORY to memoryInfo.lowMemory
        )
    }

    private fun collectNetworkInfo(): Map<String, Any> {
        val NETWORK_OPERATOR_NAME = "network_operator_name"
        val SIM_COUNTRY_ISO = "sim_country_iso"
        val IS_DATA_ENABLED = "data_enabled"
        return mapOf(
            NETWORK_OPERATOR_NAME to (telephonyInfo?.simOperator1 ?: NA),
            SIM_COUNTRY_ISO to (telephonyInfo?.simCountryISO ?: NA),
            IS_DATA_ENABLED to (telephonyInfo?.isDataEnabled ?: NA)
        )

    }

    private fun collectBatteryInfo(context: Context): Map<String, Any> {
        return Common.getBatteryInfo(context)
    }

    private fun collectLocaleInfo(): Map<String, Any> {
        val TIMEZONE = "time_zone"
        val REGION_CODE = "region_code"
        val LOCALE_DISPLAY_LANGUAGE = "locale_display_language"
        val LANGUAGE_CODE = "language_code"
        val locale = Locale.getDefault()
        return mapOf(
            TIMEZONE to TimeZone.getDefault().id,
            REGION_CODE to locale.country,
            LOCALE_DISPLAY_LANGUAGE to locale.displayLanguage,
            LANGUAGE_CODE to locale.language
        )
    }

    private fun collectScreenInfo(context: Context): Map<String, Any> {
        val displayMetrics = Common.getDisplayMetrics(context)
        val SCREEN_SCALE = "screen_scale"
        val SCREEN_SIZE = "screen_size"
        val SCREEN_RESOLUTION = "screen_resolution"
        val screenSizeWidth = displayMetrics.widthPixels / displayMetrics.xdpi
        val screenSizeHeight = displayMetrics.heightPixels / displayMetrics.ydpi
        Helpers.printLog("${displayMetrics.widthPixels} ${displayMetrics.heightPixels} $screenSizeWidth $screenSizeHeight")
        return mapOf(
            SCREEN_SCALE to displayMetrics.densityDpi,
            SCREEN_SIZE to sqrt((screenSizeWidth * screenSizeWidth) + (screenSizeHeight * screenSizeHeight)),
            SCREEN_RESOLUTION to KLPPair(displayMetrics.widthPixels, displayMetrics.heightPixels).toJson()
        )
    }

    private fun collectCPUInfo(): Map<String, Any> {
        val CPU_TYPE = "cpu_type"
        return mapOf(
            CPU_TYPE to Build.CPU_ABI
        )
    }


    private fun collectDiskInfo(): Map<String, Any> {
        return Common.getStorageInfo()
    }


    private fun collectDeviceInfo(): Map<String, Any> {
        val BOARD = "board_info"
        val BOOTLOADER = "bootloader_info"
        val PRODUCT = "product_info"
        val HARDWARE = "hardware"
        val MANUFACTURER = "manufacturer"
        val DEVICE_MODEL = "device_model"
        val DEVICE_NAME = "device_name"
        val SYSTEM_NAME = "system_name"
        val SYSTEM_VERSION = "system_version"
        val DEVICE_PRODUCT = "device_product"
        return mapOf(
            BOARD to Build.BOARD, // exynos9810
            BOOTLOADER to Build.BOOTLOADER, // N770FXXS9HXA3
            PRODUCT to Build.PRODUCT, // samsung
            HARDWARE to Build.HARDWARE, // exynos9810
            MANUFACTURER to Build.MANUFACTURER, // samsung
            DEVICE_MODEL to Build.MODEL, // SM-N770F
            DEVICE_NAME to Build.FINGERPRINT, // samsung/r7nsxx/r7:13/TP1A.220624.014/N770FXXU9HWI2:user/release-keys
            SYSTEM_NAME to DEVICE_ANDROID,
            SYSTEM_VERSION to Build.VERSION.SDK_INT,
            DEVICE_PRODUCT to Build.MANUFACTURER // Samsung
        )
    }

}