package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.app.AppOpsManager
import android.app.usage.UsageStats
import android.app.usage.UsageStatsManager
import android.content.Context
import android.os.Build
import vn.kalapa.behaviorsdk.utils.Helpers
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Date
import java.util.Locale

class KLPPackageUsageModule : IKLPApplicationModule() {
    override val key: String
        get() = "package_usage"
    val data = HashMap<String, Any>()
    override fun getRequirePermission(): Array<String> {
        return arrayOf(Manifest.permission.PACKAGE_USAGE_STATS)
    }

    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        val appOps = applicationContext.getSystemService(Context.APP_OPS_SERVICE) as AppOpsManager

        val granted = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) { // API 29+
            val mode = appOps.unsafeCheckOpNoThrow(
                AppOpsManager.OPSTR_GET_USAGE_STATS,
                android.os.Process.myUid(),
                applicationContext.packageName
            )
            mode == AppOpsManager.MODE_ALLOWED
        } else {
            try {
                @Suppress("DEPRECATION")
                val mode = appOps.checkOpNoThrow(
                    AppOpsManager.OPSTR_GET_USAGE_STATS,
                    android.os.Process.myUid(),
                    applicationContext.packageName
                )
                mode == AppOpsManager.MODE_ALLOWED
            } catch (e: Exception) {
                false // Return false if method isn't available
            }
        }
        return if (granted)
            completion()
        else mapOf()
    }

    private fun getWeeklyScreenTime(context: Context): Map<String, Int> {
        val usageStatsManager = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) { // API level 22 and above
            context.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager
        } else { // For API level 21, use the string literal directly
            context.getSystemService("usagestats") as UsageStatsManager
        }
        val calendar = Calendar.getInstance()
        val weeklyUsage = mutableMapOf<String, Int>()
        val dateFormat = SimpleDateFormat("yyyy-MM-dd", Locale.getDefault())

        // Loop for the last 7 days. Each iteration, query for one day.
        for (i in 0 until 7) {
            val endTime = calendar.timeInMillis
            calendar.add(Calendar.DAY_OF_YEAR, -1)
            val startTime = calendar.timeInMillis

            val statsList: List<UsageStats> = usageStatsManager.queryUsageStats(
                UsageStatsManager.INTERVAL_DAILY,
                startTime,
                endTime
            )

            var totalUsageForDay = 0L
            statsList.forEach { usageStats ->
                totalUsageForDay += usageStats.totalTimeInForeground
            }
            // Format the date for the day (using the start of the period)
            val dayString = dateFormat.format(Date(startTime))
            weeklyUsage[dayString] = convertMillisecondsToMinutes(totalUsageForDay)
        }
        return weeklyUsage
    }


    private fun getDailyScreenTime(context: Context): Int {
        val usageStatsManager = context.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager

        val calendar = Calendar.getInstance()
        val endTime = calendar.timeInMillis
        calendar.add(Calendar.DAY_OF_YEAR, -1) // Get stats for the last 24 hours
        val startTime = calendar.timeInMillis

        val usageStatsList: List<UsageStats> =
            usageStatsManager.queryUsageStats(UsageStatsManager.INTERVAL_DAILY, startTime, endTime)

        var totalTime: Long = 0

        for (usageStats in usageStatsList) {
            totalTime += usageStats.totalTimeInForeground
        }

        return convertMillisecondsToMinutes(totalTime)
    }

    private fun getMostUsedApps(context: Context): Map<String, Int> {
        val usageStatsManager = context.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager

        val calendar = Calendar.getInstance()
        val endTime = calendar.timeInMillis
        calendar.add(Calendar.DAY_OF_YEAR, -1)
        val startTime = calendar.timeInMillis

        val usageStatsList: List<UsageStats> =
            usageStatsManager.queryUsageStats(UsageStatsManager.INTERVAL_DAILY, startTime, endTime)

        val usageMap = mutableMapOf<String, Int>()

        for (usageStats in usageStatsList) {
            if (usageStats.totalTimeInForeground > 0) {
                val appName = APP_PACKAGES[usageStats.packageName] ?: usageStats.packageName
                usageMap[appName] = convertMillisecondsToMinutes(usageStats.totalTimeInForeground) // Convert to minutes
            }
        }

        return usageMap.entries
            .sortedByDescending { it.value } // Sort by usage time
            .associate { it.key to it.value } // Convert back to a sorted map
    }


    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        Helpers.printLog("KLPPackageUsageModule setupCollectors")
        return getPermissionAndNeededInfo(applicationContext, completion = {
            val DAILY_SCREEN_TIMES = "daily_screen_time" // Int
            val WEEKLY_SCREEN_TIMES = "weekly_screen_time" // Map<String,Int>
            val MOST_USED_APPS = "most_used_apps" // Map<String,Int>
            val dailyScreenTime = getDailyScreenTime(applicationContext)
            val weeklyScreenTime = getWeeklyScreenTime(applicationContext)
//            Helpers.printLog("KLPPackageUsageModule Daily Screen Time: $dailyScreenTime minutes")
//            Helpers.printLog("KLPPackageUsageModule weeklyScreenTimeMinutes Screen Time: $weeklyScreenTime")
            val appUsageList = getMostUsedApps(applicationContext)
//            for (app in appUsageList) {
//                Helpers.printLog("KLPPackageUsageModule App: ${app.key}, Time Used: ${app.value} minutes")
//            }
            data[key] = mapOf(
                DAILY_SCREEN_TIMES to dailyScreenTime,
                WEEKLY_SCREEN_TIMES to weeklyScreenTime,
                MOST_USED_APPS to appUsageList
            )
            Helpers.printLog("$key: ${data[key]}")
            data
        })
    }

}