package vn.kalapa.behaviorsdk.components

import android.Manifest
import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.database.Cursor
import android.os.Build
import android.provider.MediaStore
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import org.json.JSONObject
import vn.kalapa.behaviorsdk.models.KLPBehaviorObject
import vn.kalapa.behaviorsdk.models.KLPPair
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.Helpers

class KLPMediaModule : IKLPModule {
    override val key: String
        get() = "media"
    private val MEDIA_CAMERA_VIDEO_FILES = "camera_video_files"
    private val MEDIA_IMAGE_FILES_FOR_360_DAYS = "images_files_for_360_days"
    private val MEDIA_IMAGE_FILES_WITH_LOCATION = "images_files_with_location"
    private val MEDIA_IMAGES_FILES_FOR_30_DAYS = "images_files_for_30_days"
    private val MEDIA_CAMERA_IMAGES_FILE_FOR_30_DAYS = "camera_images_files_for_30_days"
    private val MEDIA_IMAGE_RESOLUTION = "image_resolution"
    private val MEDIA_CAMERA_IMAGES_FILES = "camera_images_files"
    private val MEDIA_VIDEO_FILES_COUNT = "video_files_count"
    private val MEDIA_IMAGES_FILES_COUNT = "images_files_count"
    private val MEDIA_CAMERA_VIDEO_FILES_FOR_30_DAYS = "camera_video_files_for_30_days"
    private val MEDIA_VIDEO_FILES_FOR_30_DAYS = "video_files_for_30_days"
    val data = HashMap<String, Any>()

    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        val hasPermissions = if (Build.VERSION.SDK_INT > Build.VERSION_CODES.S_V2) {
            listOf(
                Manifest.permission.READ_MEDIA_IMAGES,
                Manifest.permission.READ_MEDIA_VIDEO
            ).all { Common.checkIfPermissionGranted(applicationContext, it) }
        } else {
            Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_EXTERNAL_STORAGE)
        }
        return if (hasPermissions) completion() else mapOf()
    }

    override fun getRequirePermission(): Array<String> {
        return when {
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU -> arrayOf(
                Manifest.permission.READ_MEDIA_IMAGES,
                Manifest.permission.READ_MEDIA_VIDEO
            )
            Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q -> arrayOf(
                Manifest.permission.READ_EXTERNAL_STORAGE
            )
            else -> arrayOf(Manifest.permission.READ_EXTERNAL_STORAGE)
        }
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            val imagesGranted = Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_MEDIA_IMAGES)
            val videosGranted = Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_MEDIA_VIDEO)

            val mediaData = mutableMapOf<String, Any>()

            if (imagesGranted) {
                mediaData[MEDIA_IMAGE_FILES_FOR_360_DAYS] = getImagesFilesFor360Days(applicationContext)
                mediaData[MEDIA_IMAGE_FILES_WITH_LOCATION] = getImagesFilesWithLocation(applicationContext)
                mediaData[MEDIA_IMAGES_FILES_FOR_30_DAYS] = getImagesFilesFor30Days(applicationContext)
                mediaData[MEDIA_CAMERA_IMAGES_FILE_FOR_30_DAYS] = getCameraImagesFilesFor30Days(applicationContext)
                mediaData[MEDIA_IMAGE_RESOLUTION] = getImageResolution(applicationContext)
                mediaData[MEDIA_CAMERA_IMAGES_FILES] = getCameraImagesFilesCount(applicationContext)
                mediaData[MEDIA_IMAGES_FILES_COUNT] = getImageFilesCount(applicationContext)
            }
            if (videosGranted) {
                mediaData[MEDIA_CAMERA_VIDEO_FILES] = getVideoFilesCount(applicationContext)
                mediaData[MEDIA_VIDEO_FILES_COUNT] = getVideoFilesCount(applicationContext)
                mediaData[MEDIA_CAMERA_VIDEO_FILES_FOR_30_DAYS] = getCameraVideoFilesFor30Days(applicationContext)
                mediaData[MEDIA_VIDEO_FILES_FOR_30_DAYS] = getVideoFilesFor30Days(applicationContext)
            }

            if (mediaData.isNotEmpty()) {
                data[key] = mediaData
                return data
            } else {
                return emptyMap()
            }
        } else {
            // Android 12 or lower
            val granted = Common.checkIfPermissionGranted(applicationContext, Manifest.permission.READ_EXTERNAL_STORAGE)
            if (granted) {
                data[key] = mapOf(
                    MEDIA_CAMERA_VIDEO_FILES to getVideoFilesCount(applicationContext),
                    MEDIA_IMAGE_FILES_FOR_360_DAYS to getImagesFilesFor360Days(applicationContext),
                    MEDIA_IMAGE_FILES_WITH_LOCATION to getImagesFilesWithLocation(applicationContext),
                    MEDIA_IMAGES_FILES_FOR_30_DAYS to getImagesFilesFor30Days(applicationContext),
                    MEDIA_CAMERA_IMAGES_FILE_FOR_30_DAYS to getCameraImagesFilesFor30Days(applicationContext),
                    MEDIA_IMAGE_RESOLUTION to getImageResolution(applicationContext),
                    MEDIA_CAMERA_IMAGES_FILES to getCameraImagesFilesCount(applicationContext),
                    MEDIA_VIDEO_FILES_COUNT to getVideoFilesCount(applicationContext),
                    MEDIA_IMAGES_FILES_COUNT to getImageFilesCount(applicationContext),
                    MEDIA_CAMERA_VIDEO_FILES_FOR_30_DAYS to getCameraVideoFilesFor30Days(applicationContext),
                    MEDIA_VIDEO_FILES_FOR_30_DAYS to getVideoFilesFor30Days(applicationContext)
                )
                return data
            } else {
                return emptyMap()
            }
        }
    }

    private fun getImageFilesCount(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            null,
            null,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getVideoFilesCount(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Video.Media._ID)
        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Video.Media.EXTERNAL_CONTENT_URI,
            projection,
            null,
            null,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getCameraImagesFilesCount(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val selection = "${MediaStore.Images.Media.BUCKET_DISPLAY_NAME} = ?"
        val selectionArgs = arrayOf("Camera")

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getCameraVideoFilesCount(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Video.Media._ID)
        val selection = "${MediaStore.Video.Media.BUCKET_DISPLAY_NAME} = ?"
        val selectionArgs = arrayOf("Camera")

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Video.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getImagesFilesFor360Days(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val now = System.currentTimeMillis()
        val last360Days = now - 360L * 24 * 60 * 60 * 1000

        val selection = "${MediaStore.Images.Media.DATE_TAKEN} >= ?"
        val selectionArgs = arrayOf(last360Days.toString())

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getImagesFilesWithLocation(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val selection = "${MediaStore.Images.Media.LATITUDE} IS NOT NULL AND ${MediaStore.Images.Media.LONGITUDE} IS NOT NULL"

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            null,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getImagesFilesFor30Days(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val now = System.currentTimeMillis()
        val last30Days = now - 30L * 24 * 60 * 60 * 1000

        val selection = "${MediaStore.Images.Media.DATE_TAKEN} >= ?"
        val selectionArgs = arrayOf(last30Days.toString())

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getCameraImagesFilesFor30Days(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Images.Media._ID)
        val now = System.currentTimeMillis()
        val last30Days = now - 30L * 24 * 60 * 60 * 1000

        val selection = "${MediaStore.Images.Media.DATE_TAKEN} >= ? AND ${MediaStore.Images.Media.BUCKET_DISPLAY_NAME} = ?"
        val selectionArgs = arrayOf(last30Days.toString(), "Camera")

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getImageResolution(applicationContext: Context): List<JSONObject> {
        val projection = arrayOf(MediaStore.Images.Media.WIDTH, MediaStore.Images.Media.HEIGHT)
        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            projection,
            null,
            null,
            null
        )

        val resolutions = mutableListOf<JSONObject>()
        cursor?.use {
            while (it.moveToNext()) {
                val width = it.getInt(it.getColumnIndexOrThrow(MediaStore.Images.Media.WIDTH))
                val height = it.getInt(it.getColumnIndexOrThrow(MediaStore.Images.Media.HEIGHT))
                resolutions.add(KLPPair(width, height).toJson())
            }
        }
        return resolutions
    }


    private fun getCameraVideoFilesFor30Days(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Video.Media._ID)
        val now = System.currentTimeMillis()
        val last30Days = now - 30L * 24 * 60 * 60 * 1000

        val selection = "${MediaStore.Video.Media.DATE_TAKEN} >= ? AND ${MediaStore.Video.Media.BUCKET_DISPLAY_NAME} = ?"
        val selectionArgs = arrayOf(last30Days.toString(), "Camera")

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Video.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }

    private fun getVideoFilesFor30Days(applicationContext: Context): Int {
        val projection = arrayOf(MediaStore.Video.Media._ID)
        val now = System.currentTimeMillis()
        val last30Days = now - 30L * 24 * 60 * 60 * 1000

        val selection = "${MediaStore.Video.Media.DATE_TAKEN} >= ?"
        val selectionArgs = arrayOf(last30Days.toString())

        val cursor: Cursor? = applicationContext.contentResolver.query(
            MediaStore.Video.Media.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            null
        )
        val count = cursor?.count ?: 0
        cursor?.close()
        return count
    }
}
