package vn.kalapa.behaviorsdk.components

import android.annotation.SuppressLint
import android.app.Activity
import android.app.ActivityManager
import android.content.Context
import android.content.Intent
import android.os.Build
import org.json.JSONObject
import vn.kalapa.behaviorsdk.behavioral.KLPGlobalEventService
import vn.kalapa.behaviorsdk.managers.PeriodicTaskManager
import vn.kalapa.behaviorsdk.models.KLPBehaviorObject
import vn.kalapa.behaviorsdk.service.KLPBehaviorScore
import vn.kalapa.behaviorsdk.utils.Common.Companion.putBehaviorSet
import vn.kalapa.behaviorsdk.utils.CryptoHelpers
import vn.kalapa.behaviorsdk.utils.Helpers

class KLPBehavioralModule : IKLPModule {
    override val key: String get() = "behavior"
    override fun getRequirePermission(): Array<String> {
        return emptyArray()
    }

    override fun getPermissionAndNeededInfo(applicationContext: Context, completion: () -> Map<String, Any>): Map<String, Any> {
        return completion()
    }

    override suspend fun setupCollectors(applicationContext: Context): Map<String, Any> {
        val result = getPermissionAndNeededInfo(applicationContext, completion = {
            Helpers.printLog("KLPBehavioralModule: $behaviorList")
            mapOf(key to mapOf(BEHAVIOR_ACTION to behaviorList))
        })
        Helpers.printLog("$key - setupCollectors $result")
        return result
    }

    companion object {
        val BEHAVIOR_ACTION = "actions"

        private var shouldCollect = false
        private val behaviorList = ArrayList<JSONObject>()
        private lateinit var serviceIntent: Intent
        lateinit var applicationContext: Context
        private var periodicTaskManager: PeriodicTaskManager? = null

        @SuppressLint("StaticFieldLeak")
        lateinit var trackingActivity: Activity

        fun isApplicationContextInitialized(): Boolean {
            return this::applicationContext.isInitialized
        }

        fun isTrackingActivityInitialized(): Boolean {
            return this::trackingActivity.isInitialized
        }

        fun startTracking(activity: Activity) {
            startTracking(activity, false)
        }

        fun startTracking(activity: Activity, continuousTracking: Boolean) {
            shouldCollect = true
            if (continuousTracking && periodicTaskManager == null) {
                periodicTaskManager = PeriodicTaskManager {
                    Helpers.printLog("Periodic task is running: ${behaviorList.size} - doc_id: ${KLPBehaviorScore.getDocID()}")
                    if (behaviorList.isNotEmpty() && KLPBehaviorScore.getDocID().isNotEmpty()) {
                        val dataset: Map<String, Any> = mapOf(
                            "doc_id" to KLPBehaviorScore.getDocID(),
                            "behavior" to mapOf(BEHAVIOR_ACTION to getBehaviorList())
                        )
                        Helpers.printLog("continuousTracking behavior: ${dataset["behavior"]}")
                        CryptoHelpers().encryptDataset(KLPBehaviorScore.getAccessToken(), dataset)?.let { encryptedData ->
                            if (KLPBehaviorScore.getDocID().isNotEmpty()) {
                                putBehaviorSet(KLPBehaviorScore.getBaseURL(), KLPBehaviorScore.getAccessToken(), encryptedData.first, encryptedData.second)
                                behaviorList.clear()
                            }
                        }
                    }
                }.apply {
                    startPeriodicTask(8)
                }
            }
            startService(activity)
        }

        private fun isServiceRunning(context: Context, serviceClass: Class<*>): Boolean {
            val manager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
            for (service in manager.getRunningServices(Int.MAX_VALUE)) {
                if (service.service.className == serviceClass.name) {
                    return true
                }
            }
            return false
        }

        private fun startService(activity: Activity) {
            val serviceClass = KLPGlobalEventService::class.java
            if (isServiceRunning(activity.applicationContext, serviceClass)) {
                Helpers.printLog("Service is already running, skipping start.")
                return
            }
            Helpers.printLog("Start Tracking!!!")
            this.trackingActivity = activity
            this.applicationContext = activity.applicationContext
            serviceIntent = Intent(applicationContext, serviceClass)
            serviceIntent.putExtra("EXTRA_ACTIVITY_ROOT", true)
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    Helpers.printLog("Starting foreground service...")
//                    applicationContext.startForegroundService(serviceIntent)
                    applicationContext.startService(serviceIntent)
                } else {
                    Helpers.printLog("Starting background service...")
                    applicationContext.startService(serviceIntent)
                }
            } catch (e: Exception) {
                e.printStackTrace()
                Helpers.printLog("Error starting service: ${e.message}")
            }
        }

        private fun stopService(applicationContext: Context) {
            Helpers.printLog("Stop Service...")
            if (this::serviceIntent.isInitialized)
                applicationContext.stopService(serviceIntent)
        }

        fun clearBehaviorList() {
            behaviorList.clear()
        }

        fun stopTracking() {
            shouldCollect = false
            behaviorList.clear()
            periodicTaskManager?.stopPeriodicTask()
            periodicTaskManager = null
            if (this::serviceIntent.isInitialized && this::applicationContext.isInitialized)
                stopService(applicationContext)
            else
                Helpers.printLog("The service has already been stopped or has not even started.")
        }

        internal fun addBehavior(behavior: KLPBehaviorObject) {
            if (shouldCollect) {
                Helpers.printLog("addBehavior ${behavior.event} ${behavior.target} $behavior")
                behaviorList.add(behavior.toJson())
            }
        }

        fun addBehaviorManually(behavior: KLPBehaviorObject) {
            if (shouldCollect) {
                Helpers.printLog("addBehavior ${behavior.event} ${behavior.target} $behavior")
                behaviorList.add(behavior.toJson())
            }
        }

        private fun getBehaviorList(): List<JSONObject> {
            Helpers.printLog("getBehaviorList: ${behaviorList.size} $behaviorList")
            if (!this::serviceIntent.isInitialized && !this::applicationContext.isInitialized)
                return emptyList()
            return behaviorList.toList()
        }
    }

}
