package vn.kalapa.behaviorsdk.service

import android.Manifest
import android.app.Activity
import android.app.AppOpsManager
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import android.provider.Settings
import android.widget.Toast
import androidx.fragment.app.FragmentActivity
import kotlinx.coroutines.*
import vn.kalapa.behaviorsdk.R
import vn.kalapa.behaviorsdk.components.IKLPModule
import vn.kalapa.behaviorsdk.components.KLPBehavioralModule
import vn.kalapa.behaviorsdk.handlers.ActivateModelHandler
import vn.kalapa.behaviorsdk.handlers.SubmitDatasetHandler
import vn.kalapa.behaviorsdk.managers.PermissionRequester
import vn.kalapa.behaviorsdk.models.AppToken
import vn.kalapa.behaviorsdk.models.KLPBehaviorBannerAd
import vn.kalapa.behaviorsdk.models.SubmitDataDocID
import vn.kalapa.behaviorsdk.models.SubmitDataResponse
import vn.kalapa.behaviorsdk.utils.Common
import vn.kalapa.behaviorsdk.utils.CryptoHelpers
import vn.kalapa.behaviorsdk.utils.Helpers
import vn.kalapa.behaviorsdk.utils.isPermissionDeclared
import java.util.Collections
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

interface IKLPBehaviorScore {
    fun collect(completion: (SubmitDataResponse?) -> Unit)
    suspend fun collect(): SubmitDataResponse?
}

const val PROD_URL = "https://behavior-score.kalapa.vn"
const val DEV_URL = "https://behavior-score-dev.kalapa.vn"

class KLPBehaviorScore private constructor(
    private val activity: Activity,
    private val collectors: List<IKLPModule>,
    appKey: String,
    datasetID: String,
    baseURL: String,
) : IKLPBehaviorScore {

    companion object {
        private const val VERSION = "1.0.7.5" // Change behavior
        private var docID: String = ""
        private var accessToken: String = ""
        private var datasetID: String = ""
        private var appKey: String = ""
        private var baseURL: String = ""
        fun getDocID(): String {
            return docID
        }

        fun getSDKVersion(): String {
            return VERSION
        }

        fun getAccessToken(): String {
            return accessToken
        }

        fun getDatasetID(): String {
            return datasetID
        }

        fun setAdsLayout(activity: Activity, adView: KLPBehaviorBannerAd) {
            adView.loadAd()
            adView.registerActivityLifecycleCallbacks(activity)
        }

        fun getBaseURL(): String {
            return baseURL
        }
    }

    init {
        Companion.datasetID = datasetID
        Companion.appKey = appKey
        Companion.baseURL = baseURL
        Companion.datasetID = datasetID
    }


    private fun setupPermissionRequester(context: Context): PermissionRequester =
        (context as? FragmentActivity)?.let {
            PermissionRequester(activity,it.supportFragmentManager)
        } ?: throw IllegalStateException("Context must be an instance of FragmentActivity")

    private fun getRequestPermissionFromCollectors(): Array<String> {
        val permission = Collections.synchronizedSet(HashSet<String>()) // Supports API 21+
        for (collector in collectors) {
            for (p in collector.getRequirePermission()) {
                if (isPermissionDeclared(activity, p)) {
                    permission.add(p)
                }
            }
        }
        return permission.toTypedArray()
    }

    private suspend fun doLogin(appKey: String, url: String): String = withContext(Dispatchers.IO) {
        try {
            val isOnline = Common.isOnline(activity)
            if (!isOnline) throw IllegalStateException("No internet connection available.")
            if (appKey.isEmpty()) throw IllegalArgumentException("Invalid appKey provided.")
            val responseBody = ActivateModelHandler(activity).execute(url, appKey).get()
            Helpers.printLog("responseBody: $responseBody")
            responseBody
                ?.takeIf { it.contains("data") && it.contains("code") }
                ?.let { AppToken.fromJson(it) }
                ?.let { return@withContext it.data.access_token }
                ?: throw IllegalArgumentException("Invalid response received: $responseBody")
        } catch (e: Exception) {
            Helpers.printLog("Failed to build KLPBehaviorScore: ${e.message}")
            ""
        }
    }

    private suspend fun doCollectWorkflow(): SubmitDataResponse? = withContext(Dispatchers.IO) {
        try {
            if (accessToken.isEmpty()) {
                if (appKey.isEmpty()) return@withContext SubmitDataResponse(SubmitDataDocID(docID), 400, "Invalid appKey provided.")
                else {
                    accessToken = doLogin(appKey, baseURL)
                    if (accessToken.isEmpty())
                        return@withContext SubmitDataResponse(SubmitDataDocID(docID), 401, "Invalid response received: $accessToken")
                }
            }

            var behaviorMap: Map<String, Any>? = null
            val results = collectors.map {
                async {
                    val map = it.setupCollectors(activity) // Get the collected map

                    if (it is KLPBehavioralModule) {
                        Helpers.printLog("KLPBehavioralModule: $map")
                        behaviorMap = map // Store behaviorMap separately
                    }
                    // Return a new map without "behavior", ensuring behaviorMap is unchanged
                    map.filterKeys { key -> key != "behavior" }
                }
            }.awaitAll()
            val dataset = behaviorMap?.let { createDataset(results, it) } ?: createDataset(results, null)
            behaviorMap?.let {
                Helpers.printLog("Dataset behavior: ${dataset["behavior"]}")
            }
            Helpers.printLog("Dataset collected: sdk_version: ${dataset["sdk_version"]} - $datasetID - clientKey $accessToken")

            return@withContext CryptoHelpers().encryptDataset(accessToken, dataset)?.let { encryptedData ->
                SubmitDatasetHandler()
                    .execute(baseURL, accessToken, encryptedData.first, encryptedData.second)
                    .get()
                    .takeIf { it.isNotEmpty() }
                    ?.let { SubmitDataResponse.fromJson(it) } // Gson().fromJson(it, SubmitDataResponse::class.java)
            }
        } catch (e: Exception) {
            handleError(e)
            e.printStackTrace()
            null
        } finally {
            KLPBehavioralModule.clearBehaviorList()
        }
    }

    override fun collect(completion: (SubmitDataResponse?) -> Unit) {
        Helpers.printLog("KLPBehaviorScore collecting (callback)…")
        val handler = setupPermissionRequester(activity)
        CoroutineScope(Dispatchers.IO).launch {
            waitForPermission(handler, getRequestPermissionFromCollectors())
            val response = doCollectWorkflow()
            Helpers.printLog("submitData doc_id: ${response?.data?.doc_id}")
            response?.data?.doc_id?.also { docID = it }
            completion(response)
        }
    }

    override suspend fun collect(): SubmitDataResponse? {
        Helpers.printLog("KLPBehaviorScore collecting (suspend)…")
        val handler = setupPermissionRequester(activity)
        waitForPermission(handler, getRequestPermissionFromCollectors())
        val response = doCollectWorkflow()
        Helpers.printLog("submitData doc_id: ${response?.data?.doc_id}")
        response?.data?.doc_id?.also { docID = it }
        return response
    }

    private fun createDataset(results: List<Map<String, Any>>, behavior: Map<String, Any>?): Map<String, Any> {
        val devices = Collections.synchronizedMap(mutableMapOf<String, Any>())

        return mutableMapOf<String, Any>(
            "dataset_id" to datasetID,
            "platform" to "android",
            "sdk_version" to VERSION,
            "device" to devices.apply {
                results.forEach { devices.putAll(it) }
            }
        ).apply {
            behavior?.let {
                put("behavior", extractActions(it))
            }
        }
    }

    private fun extractActions(behavior: Map<String, Any>): Map<String, Any> {
        return (behavior["behavior"] as? Map<*, *>)?.get("actions")?.let { actions ->
            mapOf("actions" to actions)
        } ?: emptyMap()
    }


    private fun checkAndRequestUsageAccess(): Boolean {
        val appOps = activity.getSystemService(Context.APP_OPS_SERVICE) as AppOpsManager
        val granted = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) { // API 29+
            val mode = appOps.unsafeCheckOpNoThrow(
                AppOpsManager.OPSTR_GET_USAGE_STATS,
                android.os.Process.myUid(),
                activity.packageName
            )
            mode == AppOpsManager.MODE_ALLOWED
        } else {
            try {
                @Suppress("DEPRECATION")
                val mode = appOps.checkOpNoThrow(
                    AppOpsManager.OPSTR_GET_USAGE_STATS,
                    android.os.Process.myUid(),
                    activity.packageName
                )
                mode == AppOpsManager.MODE_ALLOWED
            } catch (e: Exception) {
                false // Return false if method isn't available
            }
        }

        if (!granted) {
            val manufacturer = Build.MANUFACTURER

            val usageIntent = Intent(Settings.ACTION_USAGE_ACCESS_SETTINGS)
            usageIntent.data = Uri.parse("package:" + activity.packageName)
            if (usageIntent.resolveActivity(activity.packageManager) != null) {
                activity.startActivity(usageIntent)
            } else {
                if (manufacturer.equals("OPPO", ignoreCase = true)) {
                    // Try OPPO's specific usage access settings page.
                    val oppoIntent = Intent().apply { setComponent(ComponentName("com.coloros.safecenter", "com.coloros.privacypermissionsentry.PermissionTopActivity")) }
                    if (oppoIntent.resolveActivity(activity.packageManager) != null) {
                        activity.startActivity(oppoIntent)
                    } else {
                        Helpers.printLog("Oppo can not resolve Activity")
                    }
                } else {
                    // Fallback: Open the general Settings page.
                    val settingsIntent = Intent(Settings.ACTION_SETTINGS).apply { flags = Intent.FLAG_ACTIVITY_NEW_TASK }
                    if (settingsIntent.resolveActivity(activity.packageManager) != null) {
                        Helpers.printLog("Calling settingsIntent")
                        activity.startActivity(settingsIntent)
                    } else {
                        Helpers.printLog("No Activity found to handle the Intent")
                        Toast.makeText(activity, activity.getString(R.string.you_have_to_grant_permission_manual), Toast.LENGTH_LONG).show()
                    }
                }
            }
        }
        return granted
    }


    private suspend fun waitForPermission(handler: PermissionRequester?, permissions: Array<String>) {
        if (permissions.contains(Manifest.permission.PACKAGE_USAGE_STATS)) {
            //TODO:  Need to call checkAndRequestUsageAccess and if !granted. Open intent, then wait to get back to code
            checkAndRequestUsageAccess()
        }
        suspendCoroutine<Unit> { continuation ->
            Helpers.printLog("Waiting for user to grant permission... ${permissions.contentToString()}")
            handler?.requestPermission(permissions) {
                Helpers.printLog("Permission granted!")
                continuation.resume(Unit)
            }
        }
    }

    private fun handleError(e: Exception) {
        Helpers.printLog(e)
    }

    class Builder(private val activity: Activity, private val datasetID: String, private val appKey: String) {

        private val collectors = mutableListOf<IKLPModule>()
        private var useDevEnvironment: Boolean = false
        fun switchEnvironment(useDevEnvironment: Boolean): Builder {
            this.useDevEnvironment = useDevEnvironment
            return this
        }


        fun build(): KLPBehaviorScore {
            return KLPBehaviorScore(activity, collectors, appKey, datasetID, if (useDevEnvironment) DEV_URL else PROD_URL)
        }

        fun addComponent(component: IKLPModule): Builder = apply {
            collectors.add(component)
        }
    }
}

@Suppress("DEPRECATION")
fun Context.getPackageInfo(customPackageName: String): PackageInfo =
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        packageManager.getPackageInfo(customPackageName, PackageManager.PackageInfoFlags.of(0))
    } else {
        packageManager.getPackageInfo(customPackageName, 0)
    }
