package vn.kalapa.behaviorsdk.utils

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.app.ActivityManager
import android.content.Context
import android.content.Context.TELEPHONY_SERVICE
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.os.BatteryManager
import android.os.Build
import android.os.Build.VERSION_CODES.Q
import android.os.Environment
import android.os.StatFs
import android.provider.Settings
import android.telephony.TelephonyManager
import android.text.TextUtils
import android.util.DisplayMetrics
import android.util.Log
import android.view.View
import android.view.WindowManager
import androidx.core.content.ContextCompat
import com.fingerprintjs.android.fingerprint.Fingerprinter
import com.fingerprintjs.android.fingerprint.FingerprinterFactory
import com.fingerprintjs.android.fingerprint.signal_providers.StabilityLevel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.delay
import kotlinx.coroutines.withContext
//import vn.kalapa.behaviorsdk.managers.UUIDManager
import vn.kalapa.behaviorsdk.models.KLPBehaviorObject
import vn.kalapa.behaviorsdk.models.KLPTelephonyInfo
import vn.kalapa.behaviorsdk.models.PERMISSION
import vn.kalapa.behaviorsdk.service.KLPBehaviorScore
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NA
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NETWORK_CELLULAR
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NETWORK_ETHERNET
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NETWORK_NONE
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NETWORK_OTHER
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.NETWORK_WIFI
import vn.kalapa.behaviorsdk.utils.KLPBehaviorConfig.Companion.PERMISSION_DECLINED
import java.io.BufferedReader
import java.io.File
import java.io.IOException
import java.io.InputStreamReader
import java.lang.NoClassDefFoundError
import java.net.HttpURLConnection
import java.net.URL
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class Common {
    companion object {
        private const val READ_PHONE_STATE_REQUEST_CODE = 1004
        private var isRunningOnEmulator: Boolean? = null


        fun isDeviceRooted(context: Context): Boolean {
            return isRootedByCheckRootBinaries() || isRootedByExecuteCommand() || isRootedByCheckSystemProperties() || isRootedByCheckWritablePaths() || isRootedByCheckRootBinaries() || isRootedByCheckRootPackages(context)
        }

        private fun isRootedByCheckRootBinaries(): Boolean {
            val paths = arrayOf(
                "/system/bin/su",
                "/system/xbin/su",
                "/system/app/Superuser.apk",
                "/data/local/su",
                "/data/local/xbin/su",
                "/data/local/bin/su",
                "/sbin/su",
                "/su/bin/su",
                "/system/sd/xbin/su",
                "/system/bin/busybox"
            )

            for (path in paths) {
                if (File(path).exists()) {
                    return true
                }
            }
            return false
        }

        private fun isRootedByExecuteCommand(): Boolean {
            return canExecuteCommand("/system/xbin/which su")
                    || canExecuteCommand("/system/bin/which su")
                    || canExecuteCommand("which su")
                    || canExecuteCommand("su")
        }

        // 3. Check system properties
        private fun isRootedByCheckSystemProperties(): Boolean {
            val commands = arrayOf("getprop ro.secure", "getprop ro.debuggable", "id")

            return commands.any {
                try {
                    val process = Runtime.getRuntime().exec(it)
                    val reader = BufferedReader(InputStreamReader(process.inputStream))
                    val output = reader.readLine()
                    output != null && (output.contains("0") || output.contains("root"))
                } catch (e: Exception) {
                    false
                }
            }
        }

        // 4. Check writable paths
        private fun isRootedByCheckWritablePaths(): Boolean {
            val paths = arrayOf("/system", "/sbin", "/vendor", "/etc")
            return paths.any { File(it).exists() && File(it).canWrite() }
        }

        // 5. Check for known root-related apps
        private fun isRootedByCheckRootPackages(context: Context): Boolean {
            val rootApps = listOf(
                "com.topjohnwu.magisk", "eu.chainfire.supersu", "com.koushikdutta.superuser",
                "com.thirdparty.superuser", "com.noshufou.android.su", "com.devadvance.rootcloak",
                "com.kingroot.kinguser", "com.kingo.root", "com.smedialink.oneclickroot",
                "com.z4mod.z4root", "com.ramdroid.appquarantine"
            )

            val pm: PackageManager = context.packageManager
            return rootApps.any { packageName ->
                try {
                    pm.getPackageInfo(packageName, 0)
                    true
                } catch (e: PackageManager.NameNotFoundException) {
                    false
                }
            }
        }

        private fun canExecuteCommand(command: String): Boolean {
            var executedSuccessfully: Boolean
            try {
                Runtime.getRuntime().exec(command)
                executedSuccessfully = true
            } catch (e: Exception) {
                executedSuccessfully = false
            }

            return executedSuccessfully
        }

        fun isRunningOnEmulator(): Boolean? {
            isRunningOnEmulator?.let {
                return it
            }
            // Android SDK emulator
            isRunningOnEmulator = runningOnAndroidStudioEmulator()
                    || Build.FINGERPRINT.startsWith("generic")
                    || Build.FINGERPRINT.startsWith("unknown")
                    || Build.MODEL.contains("google_sdk")
                    || Build.MODEL.contains("Emulator")
                    || Build.MODEL.contains("Android SDK built for x86")
                    || Build.MODEL.contains("VirtualBox") //bluestacks
//                    || "QC_Reference_Phone" == Build.BOARD && !"Xiaomi".equals(Build.MANUFACTURER, ignoreCase = true) //bluestacks
                    || Build.MANUFACTURER.contains("Genymotion")
                    || Build.HOST == "Build2" //MSI App Player
                    || Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic")
                    || Build.PRODUCT == "google_sdk"
                    // another Android SDK emulator check
                    || System.getProperties().getProperty("ro.kernel.qemu") == "1"
                    || Build.HARDWARE.contains("goldfish")
                    || Build.HARDWARE.contains("ranchu")
                    || Build.MANUFACTURER.contains("Genymotion")
                    || Build.PRODUCT.contains("sdk_google")
                    || Build.PRODUCT.contains("sdk")
                    || Build.PRODUCT.contains("sdk_x86")
                    || Build.PRODUCT.contains("vbox86p")
                    || Build.PRODUCT.contains("emulator")
                    || Build.PRODUCT.contains("simulator")
            return isRunningOnEmulator
        }

        private fun runningOnAndroidStudioEmulator(): Boolean {
            return Build.FINGERPRINT.startsWith("google/sdk_gphone")
                    && Build.FINGERPRINT.endsWith(":user/release-keys")
                    && Build.MANUFACTURER == "Google" && Build.PRODUCT.startsWith("sdk_gphone") && Build.BRAND == "google"
                    && Build.MODEL.startsWith("sdk_gphone")
        }

        fun getNetworkConnectionType(context: Context): String {
            try {
                val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

                return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    // For Android Marshmallow (API 23) and above
                    val network = connectivityManager.activeNetwork ?: return NETWORK_NONE
                    val activeNetwork = connectivityManager.getNetworkCapabilities(network) ?: return NETWORK_NONE

                    when {
                        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) -> NETWORK_WIFI
                        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> NETWORK_CELLULAR
                        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET) -> NETWORK_ETHERNET
                        else -> NETWORK_OTHER
                    }
                } else {
                    // For older versions
                    val activeNetworkInfo: NetworkInfo? = connectivityManager.activeNetworkInfo
                    when (activeNetworkInfo?.type) {
                        ConnectivityManager.TYPE_WIFI -> NETWORK_WIFI
                        ConnectivityManager.TYPE_MOBILE -> NETWORK_CELLULAR
                        ConnectivityManager.TYPE_ETHERNET -> NETWORK_ETHERNET
                        else -> NETWORK_OTHER
                    }
                }
            } catch (se: Exception) {
                return PERMISSION_DECLINED
            }
        }

//        fun isRoamingEnabled(context: Context): String {
//            val telephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
//            return if (requestTelephonyPermission(context))
//                telephonyManager.isNetworkRoaming.toString()
//            else KLPBehaviorConfig.PERMISSION_DECLINED
//        }


        suspend fun getFingerprint(context: Context): Map<String, String> = withContext(Dispatchers.IO) {
            try {
                val fingerprinter = FingerprinterFactory.create(context)

                // Launch all fingerprint retrievals in parallel using async
                val optimalFingerprint = async { getFingerprintAsync(fingerprinter, Fingerprinter.Version.V_6, StabilityLevel.OPTIMAL) }
                val uniqueFingerprint = async { getFingerprintAsync(fingerprinter, Fingerprinter.Version.V_6, StabilityLevel.UNIQUE) }
                val stableFingerprint = async { getFingerprintAsync(fingerprinter, Fingerprinter.Version.V_6, StabilityLevel.STABLE) }

                // Await all results
                val result = mapOf(
                    "optimal" to optimalFingerprint.await(),
                    "unique" to uniqueFingerprint.await(),
                    "stable" to stableFingerprint.await()
                )
                Helpers.printLog("Common: fp_fingerprint: $result")
                return@withContext result
            } catch (e: NoClassDefFoundError) {
                return@withContext mapOf(
                    "optimal" to "",
                    "unique" to "",
                    "stable" to ""
                )
            }

        }

        // Helper function to get fingerprint asynchronously
        private suspend fun getFingerprintAsync(
            fingerprinter: Fingerprinter,
            version: Fingerprinter.Version,
            stabilityLevel: StabilityLevel
        ): String = suspendCoroutine { continuation ->
            fingerprinter.getFingerprint(version, stabilityLevel) { fingerprint ->
                continuation.resume(fingerprint)
            }
        }

        fun isBatteryCharging(context: Context): Boolean {
            val intentFilter = IntentFilter(Intent.ACTION_BATTERY_CHANGED)
            val batteryStatus: Intent? = context.registerReceiver(null, intentFilter)

            val status = batteryStatus?.getIntExtra(BatteryManager.EXTRA_STATUS, -1) ?: -1
            return status == BatteryManager.BATTERY_STATUS_CHARGING || status == BatteryManager.BATTERY_STATUS_FULL
        }

        fun getMemoryInfo(context: Context): ActivityManager.MemoryInfo {
            val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
            val memoryInfo = ActivityManager.MemoryInfo()
            activityManager.getMemoryInfo(memoryInfo)
            return memoryInfo
        }

        fun formatSizeInGB(size: Long): String {
            var sizeInKb = size / 1000f
            if (sizeInKb < 1024) return "%.2f KB".format(sizeInKb)
            var sizeInMb = sizeInKb / 1000f
            if (sizeInMb < 1024) return "%.2f MB".format(sizeInMb)
            val sizeInGb = sizeInMb / 1000f
            return "%.2f".format(sizeInGb)
        }

        fun getStorageInfo(): Map<String, String> {
            val storageInfo = mutableMapOf<String, String>()
            val TOTAL_DISK_SPACE = "total_disk_space"
            val FREE_DISK_SPACE = "free_disk_space"

            // Internal Storage
            val internalStatFs = StatFs(Environment.getDataDirectory().path)
            val internalTotal = internalStatFs.blockCountLong * internalStatFs.blockSizeLong
            val internalAvailable = internalStatFs.availableBlocksLong * internalStatFs.blockSizeLong
            storageInfo[TOTAL_DISK_SPACE] = formatSizeInGB(internalTotal)
            storageInfo[FREE_DISK_SPACE] = formatSizeInGB(internalAvailable)

            // External Storage
            if (Environment.getExternalStorageState() == Environment.MEDIA_MOUNTED) {
                val externalStatFs = StatFs(Environment.getExternalStorageDirectory().path)
                val externalTotal = externalStatFs.blockCountLong * externalStatFs.blockSizeLong
                val externalAvailable = externalStatFs.availableBlocksLong * externalStatFs.blockSizeLong
                storageInfo[TOTAL_DISK_SPACE] = formatSizeInGB(externalTotal)
                storageInfo[FREE_DISK_SPACE] = formatSizeInGB(externalAvailable)
            }

            return storageInfo
        }

        fun getAppToken(baseUrl: String, token: String): String? {
            if (token.isEmpty() || baseUrl.isEmpty()) return null
            val url = URL("$baseUrl/api/v1/auth/login-key")
            Helpers.printLog("\nSent 'POST' request to URL : $url")
            var body = ""
            with(url.openConnection() as HttpURLConnection) {
                requestMethod = "POST"
                setRequestProperty("Content-Type", "application/json")
                setRequestProperty("Accept", "application/json")

                val jsonInputString = """{"key": "$token"}"""
                // Enable input and output streams
                doInput = true
                doOutput = true
                outputStream.use {
                    val input = jsonInputString.toByteArray(charset("utf-8"))
                    it.write(input, 0, input.size)
                }
                Helpers.printLog("\nSent 'POST' request to URL : $url; Response Code : $responseCode")
                if (responseCode == 200) {
                    val reader = BufferedReader(InputStreamReader(inputStream))
                    body = reader.readText()
                    reader.close()

//                    Helpers.printLog("Response Code: $responseCode")
//                    Helpers.printLog("Response Message: $responseMessage")
                    Helpers.printLog("Response Body: $body")
                } else {
                    Helpers.printLog("POST request not worked")
                }
                disconnect()
            }
            return body.ifEmpty { null }
        }

        suspend fun putBehaviorSet(
            baseUrl: String,
            accessToken: String,
            datasetKey: String,
            dataset: String
        ): String? = withContext(Dispatchers.IO) { // Ensures the network request runs on IO thread
            if (accessToken.isEmpty() || baseUrl.isEmpty() || dataset.isEmpty()) return@withContext null

            val url = URL("$baseUrl/api/v1/datasets/chunk")
            Helpers.printLog("\nSent 'PUT' request to URL : $url $accessToken $datasetKey \n $dataset")
            var body = ""

            try {
                with(url.openConnection() as HttpURLConnection) {
                    requestMethod = "PUT"
                    setRequestProperty("Content-Type", "application/json")
                    setRequestProperty("Accept", "application/json")
                    setRequestProperty("Authorization", accessToken)

                    val jsonInputString = """{"key": "$datasetKey","data": "$dataset"}"""
                    Helpers.printLog(jsonInputString)

                    // Enable input and output streams
                    doInput = true
                    doOutput = true

                    outputStream.use {
                        val input = jsonInputString.toByteArray(Charsets.UTF_8)
                        it.write(input, 0, input.size)
                    }

                    Helpers.printLog("\nSent 'PUT' request to URL : $url; Response Code : $responseCode")

                    if (responseCode == 200) {
                        val reader = BufferedReader(InputStreamReader(inputStream))
                        body = reader.readText()
                        reader.close()
                        Helpers.printLog("Response Body: $body")
                    } else {
                        Helpers.printLog("PUT request failed with response code: $responseCode")
                    }

                    disconnect()
                }
            } catch (e: Exception) {
                Helpers.printLog("Error in PUT request: ${e.message}")
                return@withContext null
            }

            return@withContext body.ifEmpty { null }
        }


        fun submitDataset(baseUrl: String, accessToken: String, datasetKey: String, dataset: String): String? {
            if (accessToken.isEmpty() || baseUrl.isEmpty() || datasetKey.isEmpty() || dataset.isEmpty()) return null
            val url = URL("$baseUrl/api/v1/datasets/chunk")
            Helpers.printLog("\nSent 'POST' request to URL : $url")
            var body = ""
            with(url.openConnection() as HttpURLConnection) {
                requestMethod = "POST"
                setRequestProperty("Content-Type", "application/json")
                setRequestProperty("Accept", "application/json")
                setRequestProperty("Authorization", accessToken)
                val jsonInputString = """{"key": "$datasetKey","data": "$dataset"}"""
                Helpers.printLog(jsonInputString)
                // Enable input and output streams
                doInput = true
                doOutput = true
                outputStream.use {
                    val input = jsonInputString.toByteArray(charset("utf-8"))
                    it.write(input, 0, input.size)
                }
                Helpers.printLog("\nSent 'POST' request to URL : $url; Response Code : $responseCode")
                if (responseCode == 200) {
                    val reader = BufferedReader(InputStreamReader(inputStream))
                    body = reader.readText()
                    reader.close()
                    Helpers.printLog("Response Body: $body")
                } else {
                    Helpers.printLog("POST request not worked")
                }
                disconnect()
            }
            return body.ifEmpty { null }
        }

        fun getDatasetResult(baseUrl: String, datasetKey: String): String? {
            if (baseUrl.isEmpty() || datasetKey.isEmpty()) return null
            val url = URL("$baseUrl/api/v1/insights/$datasetKey")
            Helpers.printLog("\nSent 'GET' request to URL : $url")
            var body = ""
            with(url.openConnection() as HttpURLConnection) {
                requestMethod = "GET"
                setRequestProperty("Content-Type", "application/json")
                setRequestProperty("Accept", "application/json")
                connect()
                if (responseCode == 200) {
                    val reader = BufferedReader(InputStreamReader(inputStream))
                    body = reader.readText()
                    reader.close()
                    Helpers.printLog("Response Body: $body")
                } else {
                    Helpers.printLog("POST request not worked")
                }
                disconnect()
            }
            return body.ifEmpty { null }
        }


        fun isAccessibilityEnabled(context: Context): Boolean {
            var accessibilityEnabled = 0
            try {
                accessibilityEnabled = Settings.Secure.getInt(
                    context.contentResolver,
                    Settings.Secure.ACCESSIBILITY_ENABLED
                )
            } catch (e: Settings.SettingNotFoundException) {
                e.printStackTrace()
            }

            if (accessibilityEnabled == 1) {
                val enabledServices = Settings.Secure.getString(
                    context.contentResolver,
                    Settings.Secure.ENABLED_ACCESSIBILITY_SERVICES
                )
                if (!TextUtils.isEmpty(enabledServices)) {
                    return true
                }
            }
            return false
        }

        fun getBatteryInfo(context: Context): Map<String, String> {
            val BATTERY_LEVEL = "battery_level"
            val BATTERY_STATE = "battery_state"
            val BATTERY_TEMPERATURE = "battery_temperature"
            val BATTERY_TECHNOLOGY = "battery_technology"
            val BATTERY_IS_CHARGING = "is_battery_charging"
            val BATTERY_CHARGING_TYPE = "battery_charging_type"
            val batteryInfo = mutableMapOf<String, String>()

            // Register for battery information
            val intentFilter = IntentFilter(Intent.ACTION_BATTERY_CHANGED)
            val batteryStatus: Intent? = context.registerReceiver(null, intentFilter)

            // Get battery percentage
            val level = batteryStatus?.getIntExtra(BatteryManager.EXTRA_LEVEL, -1) ?: -1
            val scale = batteryStatus?.getIntExtra(BatteryManager.EXTRA_SCALE, -1) ?: -1
            val batteryPct = (level * 100 / scale.toFloat()).toInt()

            // Check if the battery is charging or not
            val status = batteryStatus?.getIntExtra(BatteryManager.EXTRA_STATUS, -1) ?: -1
            val isCharging = status == BatteryManager.BATTERY_STATUS_CHARGING || status == BatteryManager.BATTERY_STATUS_FULL

            // Determine the charging type (USB, AC, Wireless)
            val chargePlug = batteryStatus?.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1) ?: -1
            val chargingType = when (chargePlug) {
                BatteryManager.BATTERY_PLUGGED_USB -> "USB"
                BatteryManager.BATTERY_PLUGGED_AC -> "AC_ADAPTER"
                BatteryManager.BATTERY_PLUGGED_WIRELESS -> "WIRELESS"
                else -> "NOT_CHARGING"
            }

            // Get battery health
            val health = batteryStatus?.getIntExtra(BatteryManager.EXTRA_HEALTH, -1) ?: -1
            val healthStatus = when (health) {
                BatteryManager.BATTERY_HEALTH_GOOD -> "GOOD"
                BatteryManager.BATTERY_HEALTH_OVERHEAT -> "OVERHEAT"
                BatteryManager.BATTERY_HEALTH_DEAD -> "DEAD"
                BatteryManager.BATTERY_HEALTH_OVER_VOLTAGE -> "OVER_VOLTAGE"
                BatteryManager.BATTERY_HEALTH_UNSPECIFIED_FAILURE -> "FAILURE"
                BatteryManager.BATTERY_HEALTH_COLD -> "COLD"
                else -> NA
            }

            // Get battery temperature
            val temperature = batteryStatus?.getIntExtra(BatteryManager.EXTRA_TEMPERATURE, -1)?.toFloat()?.div(10) ?: -1f

            // Get battery technology
            val technology = batteryStatus?.getStringExtra(BatteryManager.EXTRA_TECHNOLOGY) ?: NA

            // Populate the battery information
            batteryInfo[BATTERY_LEVEL] = batteryPct.toString()
            batteryInfo[BATTERY_IS_CHARGING] = isCharging.toString()
            batteryInfo[BATTERY_CHARGING_TYPE] = chargingType
            batteryInfo[BATTERY_STATE] = healthStatus
            batteryInfo[BATTERY_TEMPERATURE] = temperature.toString()
            batteryInfo[BATTERY_TECHNOLOGY] = technology
            return batteryInfo
        }

        fun getDisplayMetrics(context: Context): DisplayMetrics {
            val displayMetrics = DisplayMetrics()
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            windowManager.defaultDisplay.getMetrics(displayMetrics)
            return displayMetrics
        }

        @SuppressLint("HardwareIds")
        fun getTelephonyInfo(context: Context): KLPTelephonyInfo? {
            val klpTelephonyInfoBuilder = KLPTelephonyInfo.KLPTelephonyInfoBuilder()
            klpTelephonyInfoBuilder.withUniqueDeviceId(context)
            if (checkIfPermissionGranted(context, Manifest.permission.READ_PHONE_STATE)) {
                try {
                    val telephonyManager = context.getSystemService(TELEPHONY_SERVICE) as TelephonyManager
                    klpTelephonyInfoBuilder.withSimCountryISO(telephonyManager.simCountryIso)
                    klpTelephonyInfoBuilder.withSimOperator1(telephonyManager.simOperatorName)
                    if (Build.VERSION.SDK_INT >= Q) {
                        klpTelephonyInfoBuilder.withDataEnabled(telephonyManager.isDataEnabled)
                        klpTelephonyInfoBuilder.withIsRoamingEnabled(telephonyManager.isDataRoamingEnabled)
                    }
                } catch (exception: SecurityException) {
                    Helpers.printLog("KLP ${exception.printStackTrace()}")
                }
                return klpTelephonyInfoBuilder.build()
            } else {
                Helpers.printLog("READ_PHONE_STATE not provided")
                return klpTelephonyInfoBuilder.build()
            }
        }

        fun isOnline(): Boolean {
            val runtime = Runtime.getRuntime()
            try {
                val ipProcess = runtime.exec("/system/bin/ping -c 1 8.8.8.8")
                val exitValue = ipProcess.waitFor()
                return exitValue == 0
            } catch (e: IOException) {
                e.printStackTrace()
            } catch (e: InterruptedException) {
                e.printStackTrace()
            }
            return false
        }


        fun isOnline(context: Context): Boolean {
            var isOnline = false
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
                val capabilities =
                    connectivityManager.getNetworkCapabilities(connectivityManager.activeNetwork)
                if (capabilities != null) {
                    if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
                        Log.i("Internet", "NetworkCapabilities.TRANSPORT_CELLULAR")
                        isOnline = true
                    } else if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                        Log.i("Internet", "NetworkCapabilities.TRANSPORT_WIFI")
                        isOnline = true
                    } else if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)) {
                        Log.i("Internet", "NetworkCapabilities.TRANSPORT_ETHERNET")
                        isOnline = true
                    }
                }
            } else { // < 23
                val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
                val networkInfo = connectivityManager.activeNetworkInfo
                isOnline = (networkInfo != null && networkInfo.isConnected)
            }
            return isOnline
        }

        fun mostLikelyPhoneNumber(s: String): Boolean {
            val normalizerS = s.replace("-", "").replace(" ", "").replace("+", "")
//            val regex = """(?:\+84|0084|0|84)[235789][0-9]{1,2}[0-9]{7}(?:[^\d]+|$)""".toRegex()
            val regex = """([0-9]{9,15})""".toRegex()
            return regex.matches(normalizerS)
        }

        fun isVietnamesePhoneNumber(s: String): Boolean {
            val normalizerS = s.replace("-", "").replace(" ", "").replace("+", "")
            val regex = """(?:\+84|0084|0|84)[235789][0-9]{1,2}[0-9]{7}(?:[^\d]+|$)""".toRegex()
            return regex.matches(normalizerS)
        }

        private fun getViewId(view: View): String {
            return try {
                if (view.id != View.NO_ID) {
                    val resourcesName = view.resources.getResourceName(view.id)
                    resourcesName.substringAfterLast("/", resourcesName)
                } else {
                    ""
                }
            } catch (e: Exception) {
                "" // Return empty string if ID is not found
            }
        }

        fun getKLPFormatResourcesName(view: View): String {
            val viewId = getViewId(view)
            // Ensure superclass is not null
            val className = view.javaClass.superclass?.simpleName ?: view.javaClass.simpleName
            return if (viewId.isNotEmpty()) "$className('$viewId')" else className
        }

        fun getPermissionGrantedList(context: Context): Map<String, Boolean> {
            return mapOf(
//                PERMISSION.SMS to checkIfPermissionGranted(context, Manifest.permission.READ_SMS),
//                PERMISSION.CALL to checkIfPermissionGranted(context, Manifest.permission.READ_CALL_LOG),
//                PERMISSION.MEDIA to checkIfPermissionGranted(context, if (Build.VERSION.SDK_INT > Build.VERSION_CODES.S_V2) Manifest.permission.READ_MEDIA_IMAGES else Manifest.permission.READ_EXTERNAL_STORAGE),
                PERMISSION.CALENDAR to checkIfPermissionGranted(context, Manifest.permission.READ_CALENDAR),
                PERMISSION.CONTACTS to checkIfPermissionGranted(context, Manifest.permission.READ_CONTACTS),
                PERMISSION.BLUETOOTH to checkIfPermissionGranted(context, Manifest.permission.BLUETOOTH),
                PERMISSION.DEVICE to checkIfPermissionGranted(context, Manifest.permission.READ_PHONE_STATE)
            )
        }

        fun checkIfPermissionGranted(context: Context, permission: String): Boolean {
            return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED
        }


        fun getArrayOfPermission(): Array<String> {
            return if (Build.VERSION.SDK_INT > Build.VERSION_CODES.S_V2)
                arrayOf(
                    Manifest.permission.READ_CONTACTS,
                    Manifest.permission.READ_CALENDAR,
                    Manifest.permission.READ_PHONE_STATE,
//                    Manifest.permission.BLUETOOTH,
//                    Manifest.permission.BLUETOOTH_ADMIN,
                    Manifest.permission.ACCESS_FINE_LOCATION,
                    Manifest.permission.ACCESS_COARSE_LOCATION,
//                    Manifest.permission.READ_SMS,
                    Manifest.permission.RECEIVE_SMS,
//                    Manifest.permission.READ_CALL_LOG,
                    // Different
//                    Manifest.permission.READ_MEDIA_IMAGES,
//                    Manifest.permission.READ_MEDIA_VIDEO,
//                    Manifest.permission.READ_MEDIA_AUDIO,
                )
            else
                arrayOf(
                    Manifest.permission.READ_CONTACTS,
                    Manifest.permission.READ_CALENDAR,
                    Manifest.permission.READ_PHONE_STATE,
//                    Manifest.permission.BLUETOOTH,
//                    Manifest.permission.BLUETOOTH_ADMIN,
                    Manifest.permission.ACCESS_FINE_LOCATION,
                    Manifest.permission.ACCESS_COARSE_LOCATION,
//                    Manifest.permission.READ_SMS,
                    Manifest.permission.RECEIVE_SMS,
//                    Manifest.permission.READ_CALL_LOG,
                    // Different
//                    Manifest.permission.READ_EXTERNAL_STORAGE
                )
        }
    }
}


