package vn.kalapa.ekyc.liveness

import android.content.Context
import android.graphics.Bitmap
import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import vn.kalapa.ekyc.managers.KLPFaceDetectorListener
import vn.kalapa.ekyc.utils.Common
import vn.kalapa.ekyc.utils.Helpers
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.TimeUnit

class LivenessHandler(
    private val context: Context,
    private val livenessSessionType: Common.LIVENESS_VERSION,
    private val faceDetectorListener: KLPFaceDetectorListener,
    private val rotationAngle: Int = 0
) {
    companion object {
        private const val LIVENESS_MAX_TIME = 60_000L // 1 minute in milliseconds
        private const val PROCESS_INTERVAL = 33L // ~30 FPS processing rate
    }

    private var livenessSession = LivenessSession(livenessSessionType)
    private var isStop = false
    private var startTime = System.currentTimeMillis()
    private var lastProcessTime = 0L
    private var sessionStatus: LivenessSessionStatus = LivenessSessionStatus.UNVERIFIED
    private var sessionAction: String = ""

    private val processingQueue = LinkedBlockingQueue<Pair<Bitmap, Bitmap>>(2)
    private val processingScope = CoroutineScope(Dispatchers.Default + SupervisorJob())

    init {
        startProcessingWorker()
    }

    private fun startProcessingWorker() {
        processingScope.launch {
            while (isActive) {
                try {
                    val (frame, image) = processingQueue.poll(100, TimeUnit.MILLISECONDS) ?: continue
                    processFrameInternal(frame, image)
                } catch (e: Exception) {
                    Log.e("LivenessHandler", "Error processing frame: ${e.message}")
                }
            }
        }
    }

    fun stop() {
        isStop = true
        processingScope.cancel()
        processingQueue.clear()
    }

    fun renewSession() {
        processingQueue.clear()
        livenessSession.renewSession(livenessSessionType)
        sessionStatus = livenessSession.sessionStatus
        startTime = System.currentTimeMillis()
        lastProcessTime = 0L
        isStop = false
    }

    fun processSession(frame: Bitmap, image: Bitmap) {
        val currentTime = System.currentTimeMillis()

        if (currentTime - lastProcessTime < PROCESS_INTERVAL) {
            return
        }

        if (processingQueue.size < 2) {
            processingQueue.offer(frame to image)
        }
    }

    private suspend fun processFrameInternal(frame: Bitmap, image: Bitmap) {
        val currentTime = System.currentTimeMillis()
        lastProcessTime = currentTime

        val isExpired = currentTime - startTime > LIVENESS_MAX_TIME

        if (isStop || livenessSession.isFinished() || isExpired) {
            handleSessionEnd(frame, isExpired)
            return
        }

        withContext(Dispatchers.Default) {
            livenessSession.process(frame, image, rotationAngle, faceDetectorListener)
            sessionStatus = livenessSession.sessionStatus
            sessionAction = livenessSession.getCurrentAction()
        }
    }

    private fun handleSessionEnd(frame: Bitmap, isExpired: Boolean) {
        sessionStatus = livenessSession.sessionStatus
        sessionAction = livenessSession.getCurrentAction()

        when {
            livenessSession.isFinished() && sessionStatus == LivenessSessionStatus.VERIFIED -> {
                if (livenessSession.gotTypicalFace) {
                    faceDetectorListener.onFaceDetected(
                        livenessSession.typicalFrame,
                        livenessSession.typicalFace
                    )
                }
            }
            livenessSession.isFinished() -> {
                faceDetectorListener.onFaceDetected(frame)
            }
            sessionStatus == LivenessSessionStatus.EXPIRED || isExpired -> {
                faceDetectorListener.onExpired()
            }
        }
    }
    
    fun release() {
        livenessSession.cleanup()
        stop()
        processingQueue.clear()
    }
}
