package vn.kalapa.ekyc.activity

import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Color
import android.os.Build
import android.os.Bundle
import android.util.DisplayMetrics
import androidx.appcompat.app.AppCompatActivity
import vn.kalapa.R
import vn.kalapa.ekyc.DialogListener
import vn.kalapa.ekyc.KalapaSDK
import vn.kalapa.ekyc.KalapaSDKCallback
import vn.kalapa.ekyc.KalapaSDKMediaType
import vn.kalapa.ekyc.KalapaSDKResultCode
import vn.kalapa.ekyc.capturesdk.CameraXAutoCaptureActivity
import vn.kalapa.ekyc.capturesdk.CameraXMRZActivity
import vn.kalapa.ekyc.capturesdk.CameraXPassportActivity
import vn.kalapa.ekyc.codescanner.ScanQrCodeActivity
import vn.kalapa.ekyc.iproov.IProovFaceVerificationActivity
import vn.kalapa.ekyc.managers.KLPLanguageManager
import vn.kalapa.ekyc.nfcsdk.activities.NFCActivity
import vn.kalapa.ekyc.utils.Helpers
import vn.kalapa.ekyc.utils.KALAPA_LOG_ACTION
import vn.kalapa.ekyc.utils.KALAPA_LOG_LEVEL
import vn.kalapa.ekyc.utils.LogUtil.logEndSession
import vn.kalapa.ekyc.utils.LogUtil.logEvent
import vn.kalapa.ekyc.views.ProgressView
import java.io.BufferedReader
import java.io.File
import java.io.InputStreamReader
import java.lang.ref.WeakReference
import java.util.Locale


abstract class BaseActivity : AppCompatActivity(), KalapaSDKCallback {
    lateinit var SCREEN_ID: String
    var extraInfo: String = ""

    private val entryTime = System.currentTimeMillis()

    companion object {
        init {
            System.loadLibrary("envi")
        }
    }

    open fun getIntentData() {

    }

    private fun getScreenIDFromActivity(activity: Activity): String {
        return when (val activityName = activity.javaClass.simpleName) {
            CameraXAutoCaptureActivity::class.java.simpleName -> if (extraInfo == KalapaSDKMediaType.BACK.name) "KLP_SCREEN_CAPTURE_BACK" else "KLP_SCREEN_CAPTURE_FRONT"
            CameraXPassportActivity::class.java.simpleName -> "KLP_SCREEN_PASSPORT"
            NFCActivity::class.java.simpleName -> "KLP_SCREEN_NFC"
            CameraXMRZActivity::class.java.simpleName -> "KLP_SCREEN_MRZ"
            CameraXSelfieActivity::class.java.simpleName -> "KLP_SCREEN_LIVENESS"
            ConfirmActivity::class.java.simpleName -> "KLP_SCREEN_CONFIRM"
            ScanQrCodeActivity::class.java.simpleName -> "KLP_SCREEN_SCAN_QR"
            IProovFaceVerificationActivity::class.java.simpleName -> "KLP_SCREEN_STRICT_LIVENESS"
            else -> return activityName
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        Helpers.printLog("BaseActivity ${this::class.java.simpleName} - Locale: ${Locale.getDefault()} - ${KalapaSDK.config.language}")
        KLPLanguageManager.setLocale(this@BaseActivity, KalapaSDK.config.language)
        Helpers.printLog("BaseActivity ${this::class.java.simpleName} - Locale: ${Locale.getDefault()} - ${KalapaSDK.config.language}")
        getIntentData()
        SCREEN_ID = getScreenIDFromActivity(activity = this)
        if (this::class.java.simpleName == CameraXAutoCaptureActivity::class.java.simpleName)
            SCREEN_ID = getScreenIDFromActivity(activity = this)
        if (KalapaSDK.isConfigInitialized()) {
            window.statusBarColor = Color.parseColor(KalapaSDK.config.backgroundColor)
            window.navigationBarColor = Color.parseColor(KalapaSDK.config.backgroundColor)
        }
        logEvent(this, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.ENTER_SCREEN, SCREEN_ID, mapOf("entry_time" to "$entryTime"))
    }

    override fun onDestroy() {
        super.onDestroy()
        logEvent(this, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.EXIT_SCREEN, SCREEN_ID, mapOf("exceed_time" to "${System.currentTimeMillis() - entryTime}"))
    }

    override fun sendExpired() {
        ProgressView.hideProgress()
        val weakActivity = WeakReference(this).get()
        weakActivity?.let { logEvent(it, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.SESSION_EXPIRED, SCREEN_ID) }
        Helpers.showDialog(this,
            KLPLanguageManager.get(resources.getString(R.string.klp_error_occurred_title)),
            KLPLanguageManager.get(resources.getString(R.string.klp_error_timeout)),
            KLPLanguageManager.get(resources.getString(R.string.klp_button_confirm)),
            KLPLanguageManager.get(resources.getString(R.string.klp_button_cancel)),
            null,
            false,
            object : DialogListener {
                override fun onYes() {
                    weakActivity?.let { logEvent(it, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.SESSION_EXPIRED_TAP_RETRY, SCREEN_ID) }
                    KalapaSDK.handler.onExpired()
                    finish()
                }

                override fun onNo() {
                    weakActivity?.let {
                        logEvent(it, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.SESSION_EXPIRED_TAP_GIVE_UP, SCREEN_ID)
                        logEndSession(it, KALAPA_LOG_ACTION.EKYC_EXPIRED, SCREEN_ID)
                    }
                    finish()
                }

            }
        )

    }

    private fun onEmulatorDetected() {
        // Something
        Helpers.showDialog(
            this,
            KLPLanguageManager.get(resources.getString(R.string.klp_error_occurred_title)),
            KLPLanguageManager.get(resources.getString(R.string.klp_error_emulator)),
            R.drawable.frowning_face,
            false
        ) {
            logEvent(this, KALAPA_LOG_LEVEL.WARN, KALAPA_LOG_ACTION.EMULATOR_DETECTED, SCREEN_ID)
            KalapaSDK.handler.onError(KalapaSDKResultCode.EMULATOR_DETECTED)
            finish()
        }
    }

    private fun onVirtualCameraDetected() {
        Helpers.printLog("onVirtualCameraDetected virtual camera")
    }

    private var enterScreen = false
    override fun onResume() {
        if (isRunningOnEmulator() == true) {
            Helpers.printLog("isRunningOnEmulator Emulator detected!")
            if (KalapaSDK.isConfigInitialized() && KalapaSDK.config.baseURL.contains("dev") && KalapaSDK.config.baseURL.contains(
                    "kalapa"
                )
            ) {
                Helpers.printLog("isRunningOnEmulator still accept when dev")
            } else
                onEmulatorDetected()
        }
        if (isPathReallyExist(filesDir.absolutePath) < 0) {
            Helpers.printLog("onVirtualCameraDetected fake camera")
            onEmulatorDetected()
        } else Helpers.printLog("onVirtualCameraDetected real camera")
        if (!enterScreen) {
            enterScreen = true
        }
        super.onResume()
    }

    override fun attachBaseContext(newBase: Context?) {
        val configuration = newBase!!.resources.configuration
        configuration.densityDpi = DisplayMetrics.DENSITY_DEVICE_STABLE
        applyOverrideConfiguration(configuration)
        super.attachBaseContext(newBase)
    }


    private var isRunningOnEmulator: Boolean? = null

    private fun isDeviceRooted(): Boolean {
        return isRootedByCheckRootBinaries() || isRootedByExecuteCommand() || isRootedByCheckSystemProperties() || isRootedByCheckWritablePaths() || isRootedByCheckRootBinaries()
    }

    private fun isRootedByCheckRootBinaries(): Boolean {
        val paths = arrayOf(
            "/system/bin/su",
            "/system/xbin/su",
            "/system/app/Superuser.apk",
            "/data/local/su",
            "/data/local/xbin/su",
            "/data/local/bin/su",
            "/sbin/su",
            "/su/bin/su",
            "/system/sd/xbin/su",
            "/system/bin/busybox"
        )

        for (path in paths) {
            if (File(path).exists()) {
                return true
            }
        }
        return false
    }

    private fun isRootedByExecuteCommand(): Boolean {
        return canExecuteCommand("/system/xbin/which su")
                || canExecuteCommand("/system/bin/which su")
                || canExecuteCommand("which su")
                || canExecuteCommand("su")
    }

    // 3. Check system properties
    private fun isRootedByCheckSystemProperties(): Boolean {
        val commands = arrayOf("getprop ro.secure", "getprop ro.debuggable", "id")

        return commands.any {
            try {
                val process = Runtime.getRuntime().exec(it)
                val reader = BufferedReader(InputStreamReader(process.inputStream))
                val output = reader.readLine()
                output != null && (output.contains("0") || output.contains("root"))
            } catch (e: Exception) {
                false
            }
        }
    }

    // 4. Check writable paths
    private fun isRootedByCheckWritablePaths(): Boolean {
        val paths = arrayOf("/system", "/sbin", "/vendor", "/etc")
        return paths.any { File(it).exists() && File(it).canWrite() }
    }

    // 5. Check for known root-related apps
    private fun isRootedByCheckRootPackages(): Boolean {
        val rootApps = listOf(
            "com.topjohnwu.magisk", "eu.chainfire.supersu", "com.koushikdutta.superuser",
            "com.thirdparty.superuser", "com.noshufou.android.su", "com.devadvance.rootcloak",
            "com.kingroot.kinguser", "com.kingo.root", "com.smedialink.oneclickroot",
            "com.z4mod.z4root", "com.ramdroid.appquarantine"
        )

        val pm: PackageManager = this.packageManager
        return rootApps.any { packageName ->
            try {
                pm.getPackageInfo(packageName, 0)
                true
            } catch (e: PackageManager.NameNotFoundException) {
                false
            }
        }
    }

    private fun canExecuteCommand(command: String): Boolean {
        var executedSuccessfully: Boolean
        try {
            Runtime.getRuntime().exec(command)
            executedSuccessfully = true
        } catch (e: Exception) {
            executedSuccessfully = false
        }

        return executedSuccessfully
    }

    private fun isRunningOnEmulator(): Boolean? {
        isRunningOnEmulator?.let {
            return it
        }
        // Android SDK emulator
        isRunningOnEmulator = runningOnAndroidStudioEmulator()
                || Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.startsWith("unknown")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.MODEL.contains("VirtualBox") //bluestacks
                || "QC_Reference_Phone" == Build.BOARD && !"Xiaomi".equals(
            Build.MANUFACTURER,
            ignoreCase = true
        ) //bluestacks
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.HOST == "Build2" //MSI App Player
                || Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic")
                || Build.PRODUCT == "google_sdk"
                // another Android SDK emulator check
                || System.getProperties().getProperty("ro.kernel.qemu") == "1"
                || Build.HARDWARE.contains("goldfish")
                || Build.HARDWARE.contains("ranchu")
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.PRODUCT.contains("sdk_google")
                || Build.PRODUCT.contains("sdk")
                || Build.PRODUCT.contains("sdk_x86")
                || Build.PRODUCT.contains("vbox86p")
                || Build.PRODUCT.contains("emulator")
                || Build.PRODUCT.contains("simulator")
        return isRunningOnEmulator
    }

    private fun runningOnAndroidStudioEmulator(): Boolean {
        return Build.FINGERPRINT.startsWith("google/sdk_gphone")
                && Build.FINGERPRINT.endsWith(":user/release-keys")
                && Build.MANUFACTURER == "Google" && Build.PRODUCT.startsWith("sdk_gphone") && Build.BRAND == "google"
                && Build.MODEL.startsWith("sdk_gphone")
    }

    external fun isPathReallyExist(path: String?): Int

    open fun showEndEKYC(onYesThen: (() -> Unit)? = null) {
        Helpers.showEndKYC(this, SCREEN_ID, object : DialogListener {
            override fun onYes() {
                KalapaSDK.handler.onError(KalapaSDKResultCode.USER_LEAVE)
                logEndSession(this@BaseActivity, KALAPA_LOG_ACTION.EKYC_LEAVE, SCREEN_ID)
                onYesThen?.invoke()
                finish()
            }

            override fun onNo() {
            }
        })
    }
}