package vn.kalapa.ekyc.iproov.data_source

import com.google.gson.GsonBuilder
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Response
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import vn.kalapa.ekyc.iproov.data.request.IProovTokenRequest
import vn.kalapa.ekyc.iproov.data.request.IProovValidateRequest
import vn.kalapa.ekyc.iproov.data.service.IProovService
import vn.kalapa.ekyc.utils.IProovConstant
import java.util.concurrent.TimeUnit

object IProovApi {
    private val gson = GsonBuilder().setLenient().create()

    private fun createIProovService(): IProovService {
        val client = OkHttpClient.Builder()
            .addInterceptor { chain ->
                val originalRequest = chain.request()
                val newRequest = originalRequest.newBuilder()
                    .build()
                chain.proceed(newRequest)
            }
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .writeTimeout(30, TimeUnit.SECONDS)
            .addInterceptor(HttpLoggingInterceptor().apply {
                level = HttpLoggingInterceptor.Level.BODY
            })
            .build()

        val retrofit = Retrofit.Builder()
            .baseUrl(IProovConstant.FUEL_URL)
            .client(client)
            .addConverterFactory(GsonConverterFactory.create(gson))
            .build()
        return retrofit.create(IProovService::class.java)
    }

    private val iProovService by lazy {
        createIProovService()
    }

    suspend fun getClaimToken(
        userId: String,
        resource: String,
    ) = safeApiCall {
        iProovService.getClaimToken(
            IProovTokenRequest(
                apiKey = IProovConstant.API_KEY,
                secret = IProovConstant.SECRET,
                resource = resource,
                assuranceType = "genuine_presence",
                userId = userId,
                riskProfile = ""
            )
        )
    }

    suspend fun validateClaimToken(token: String, userId: String, client: String) = safeApiCall {
        iProovService.validateClaimToken(
            IProovValidateRequest(
                apiKey = IProovConstant.API_KEY,
                secret = IProovConstant.SECRET,
                token = token,
                userId = userId,
                riskProfile = "",
                client = client,
                activate = true
            )
        )
    }

    private suspend fun <T> safeApiCall(apiCall: suspend () -> Response<T>): Result<T> =
        withContext(Dispatchers.IO) {
            try {
                val response = apiCall()
                if (response.isSuccessful) {
                    Result.success(
                        response.body() ?: throw ApiException(
                            response.code(),
                            response.message()
                        )
                    )
                } else {
                    Result.failure(ApiException(response.code(), response.message()))
                }
            } catch (e: Exception) {
                Result.failure(e)
            }
        }
}

internal class ApiException(val code: Int, override val message: String) : Exception()
