package vn.kalapa.ekyc.liveness.models

import android.graphics.Bitmap
import android.graphics.PointF
import com.google.mlkit.vision.face.Face
import vn.kalapa.ekyc.liveness.InputFace
import vn.kalapa.ekyc.liveness.LivenessSession
import vn.kalapa.ekyc.utils.Helpers
import kotlin.math.max
import kotlin.math.min
import kotlin.math.pow
import kotlin.math.sqrt


enum class LivenessActionStatus {
    FAILED, SUCCESS, TIMEOUT
}

enum class LivenessFacePosition {
    LEFT, RIGHT, STRAIGHT, UP, DOWN, TILT_LEFT, TILT_RIGHT, NOT_DETERMINE
}


abstract class LivenessAction {


    lateinit var TAG: String
    val MAX_NO_FACE_FRAME = 10
    var startTime: Long = System.currentTimeMillis()
    lateinit var frame: Bitmap
    lateinit var cropFrame: Bitmap
    var isBreakAction = false
    var nMillis = 1500 // Time to success this challenge
    var currentActionStatus: LivenessActionStatus = LivenessActionStatus.FAILED
    var timeout = 30
    val REQUIRED_STABLE_FRAME_COUNT = 2
    val STABILITY_THRESHOLD = 0.001
    protected val recentFaceSizes = ArrayDeque<Double>(REQUIRED_STABLE_FRAME_COUNT)
    private var lastCenterPoint: PointF? = null
    private val MAX_MOVEMENT_BETWEEN_FRAMES = 0.01
    protected var stableFrameCount: Int = 0

    protected fun isFacesReadyToProcess(faces: List<InputFace>): Boolean {
        if (faces.isEmpty()) {
            Helpers.printLog("isFacesReadyToProcess: No faces detected")
            resetState()
            return false
        }

        val faceMetrics = calculateFaceMetrics(faces.last())

        if (!checkSmoothMovement(faceMetrics.centerPoint)) {
            Helpers.printLog("isFacesReadyToProcess: Sudden movement detected")
            resetState()
            return false
        }

        lastCenterPoint = faceMetrics.centerPoint

        recentFaceSizes.add(faceMetrics.normalizedSize)
        if (recentFaceSizes.size > REQUIRED_STABLE_FRAME_COUNT) {
            recentFaceSizes.removeFirst()
            return true
        }
        return false
    }

    data class FaceMetrics(
        val normalizedSize: Double,
        val centerPoint: PointF,
    )


    private fun checkSmoothMovement(currentCenter: PointF): Boolean {
        val lastCenter = lastCenterPoint ?: return true

        val distance = sqrt(
            (currentCenter.x - lastCenter.x).pow(2) +
                    (currentCenter.y - lastCenter.y).pow(2)
        )

        return distance <= MAX_MOVEMENT_BETWEEN_FRAMES
    }

    protected fun checkStability(sizes: Collection<Double>): Boolean {
        val mean = sizes.average()
        val variance = sizes.map { (it - mean) * (it - mean) }.average()
        return variance < STABILITY_THRESHOLD
    }


    private fun resetState() {
        stableFrameCount = 0
        recentFaceSizes.clear()
        lastCenterPoint = null
    }

    companion object {

        @JvmStatic
        protected var FACE_AREA_TOO_SMALL_THRESHOLD = 0.03 // 0.3 in each dimension

        @JvmStatic
        protected var FACE_AREA_SMALL_THRESHOLD = 0.16 // 0.4  in each dimension

        @JvmStatic
        protected var FACE_AREA_SMALL_ENOUGH_THRESHOLD = 0.2025  // 0.45 in each dimension\

        @JvmStatic
        protected var FACE_AREA_BIG_ENOUGH_THRESHOLD = 0.36 // 0.6 in each dimension

        @JvmStatic
        protected var FACE_AREA_BIG_THRESHOLD = 0.4225  // 0.65  in each dimension

        @JvmStatic
        protected var FACE_AREA_TOO_BIG_THRESHOLD = 0.64 // 0.8 in each dimension

        @JvmStatic
        protected fun checkProximity(avgFacesSize: Double, from: Double, to: Double): Boolean =
            avgFacesSize > from && avgFacesSize < to

        protected fun calculateFaceMetrics(face: InputFace): FaceMetrics {
            val bounds = face.face.boundingBox

            val faceArea = bounds.width() * bounds.height()
            val frameArea = face.frameWidth * face.frameHeight
            val normalizedSize = faceArea.toDouble() / frameArea

            val centerPoint = PointF(
                bounds.centerX().toFloat() / face.frameWidth,
                bounds.centerY().toFloat() / face.frameHeight
            )

            return FaceMetrics(normalizedSize, centerPoint)
        }

        fun isFaceTooSmall(face: InputFace): Boolean {
            val faceMetrics = calculateFaceMetrics(face)
            return faceMetrics.normalizedSize < FACE_AREA_TOO_SMALL_THRESHOLD
        }


        fun isFaceTooBig(face: InputFace): Boolean {
            val faceMetrics = calculateFaceMetrics(face)
            return faceMetrics.normalizedSize > FACE_AREA_TOO_BIG_THRESHOLD
        }

        fun isFaceSizeInRange(face: InputFace): Boolean {
            val faceMetrics = calculateFaceMetrics(face)
            return faceMetrics.normalizedSize > FACE_AREA_SMALL_ENOUGH_THRESHOLD && faceMetrics.normalizedSize < FACE_AREA_BIG_ENOUGH_THRESHOLD
        }


        fun isFaceLookStraight(face: Face): Boolean {
            val rotY = face.headEulerAngleY // Trái phải
            val rotZ = face.headEulerAngleZ // Xa Gần
            val rotX = face.headEulerAngleX // Trên dưới
            val faceNotStraight =
                rotZ < -15 || rotZ > 15 || rotX < -15 || rotX > 15 || rotY < -15 || rotY > 15
            return !faceNotStraight
        }

        fun isFaceMarginRight(
            face: Face,
            frameWidth: Int,
            frameHeight: Int,
        ): Boolean {
            val bounds = face.boundingBox
            val isTooCloseToVertical =
                bounds.top <= frameHeight * 0.05f || bounds.top <= 50 || bounds.bottom >= frameHeight * 0.95f || bounds.bottom >= frameHeight - 25
            val isTooCloseToHorizontal =
                bounds.left <= frameWidth * 0.05f || bounds.left <= 25 || bounds.right >= frameWidth * 0.95f || bounds.right >= frameWidth - 25
            val isFaceMarginRight = !isTooCloseToHorizontal && !isTooCloseToVertical
            if (!isFaceMarginRight)
                Helpers.printLog("OptimizedLivenessSession isFaceMarginRight: isTooCloseToVertical: $isTooCloseToVertical  $isTooCloseToHorizontal")
            return isFaceMarginRight
        }

    }

    fun getFacePosition(face1: Face, face2: Face): LivenessFacePosition {
        val rotY = face1.headEulerAngleY // Trái phải. Trái: 40. Phải -40
        val rotZ = face1.headEulerAngleZ // Nghiêng trái phải. Nghiêng trái: -30 Nghiêng phải 30
        val rotX = face1.headEulerAngleX // Trên dưới Trên >25. Dưới: -20
        val face1NotStraight =
            rotZ < -15 || rotZ > 15 || rotX < -15 || rotX > 15 || rotY < -15 || rotY > 15
        val rotY1 = face2.headEulerAngleY // Trái phải. Trái: 40. Phải -40
        val rotZ1 = face2.headEulerAngleZ // Nghiêng trái phải. Nghiêng trái: -30 Nghiêng phải 30
        val rotX1 = face2.headEulerAngleX // Trên dưới Trên >25. Dưới: -20
        val face2NotStraight =
            rotZ1 < -15 || rotZ1 > 15 || rotX1 < -15 || rotX1 > 15 || rotY1 < -15 || rotY1 > 15
//        Helpers.printLog("getFacePosition Face1: X $rotX Y $rotY Z $rotZ - Face2: X $rotX1 Y $rotY1 Z $rotZ1")
        return when {
            !face1NotStraight && !face2NotStraight -> LivenessFacePosition.STRAIGHT
            min(rotY, rotY1) > 15 -> LivenessFacePosition.LEFT
            max(rotY, rotY1) < -15 -> LivenessFacePosition.RIGHT
            min(rotX, rotX1) > 15 -> LivenessFacePosition.UP
            max(rotX, rotX1) < -15 -> LivenessFacePosition.DOWN
            min(rotZ, rotZ1) > 25 -> LivenessFacePosition.TILT_RIGHT
            max(rotZ, rotZ1) < -25 -> LivenessFacePosition.TILT_LEFT
            else -> return LivenessFacePosition.NOT_DETERMINE
        }
    }


    fun getFacePosition(face1: Face): LivenessFacePosition {
        val rotY = face1.headEulerAngleY // Trái phải. Trái: 40. Phải -40
        val rotZ = face1.headEulerAngleZ // Nghiêng trái phải. Nghiêng trái: -30 Nghiêng phải 30
        val rotX = face1.headEulerAngleX // Trên dưới Trên >25. Dưới: -20
        val face1NotStraight =
            rotZ < -15 || rotZ > 15 || rotX < -15 || rotX > 15 || rotY < -15 || rotY > 15
//        Helpers.printLog("getFacePosition Face1: X $rotX Y $rotY Z $rotZ")
        val facePosition = when {
            !face1NotStraight -> LivenessFacePosition.STRAIGHT
            rotY > 25 -> LivenessFacePosition.LEFT
            rotY < -25 -> LivenessFacePosition.RIGHT
            rotX > 15 -> LivenessFacePosition.UP
            rotX < -10 -> LivenessFacePosition.DOWN
            rotZ > 15 -> LivenessFacePosition.TILT_RIGHT
            rotZ < -15 -> LivenessFacePosition.TILT_LEFT
            else -> LivenessFacePosition.NOT_DETERMINE
        }
//        Helpers.printLog("Face Position: $facePosition")
        return facePosition
    }


    fun isEyesClosed(face: Face): Boolean {
        return face.rightEyeOpenProbability == null || face.rightEyeOpenProbability!! < 0.1f || face.leftEyeOpenProbability == null || face.leftEyeOpenProbability!! < 0.1f
    }


    abstract fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus

    fun process(session: LivenessSession): LivenessActionStatus {
        if (isBreakAction) {
//            Helpers.printLog("${this.javaClass.kotlin.simpleName} is Break Action. Returning Success")
            return LivenessActionStatus.SUCCESS
        }
        if (isNotEnoughFrame()) {
//            Helpers.printLog("${this.javaClass.kotlin.simpleName} is not enough frame...")
            currentActionStatus = LivenessActionStatus.FAILED
            return currentActionStatus
        }
        if (System.currentTimeMillis() - startTime > timeout * 1000) {
            currentActionStatus = LivenessActionStatus.TIMEOUT
            return currentActionStatus
        }
        val faces: List<InputFace> = getFacesByTime(session, nMillis)
        return if (faces.size < 2) {
            if (nMillis == 0 && faces.size == 1) {
                currentActionStatus = individualProcess(session, faces)
                currentActionStatus
            } else {
//                Helpers.printLog("${this.javaClass.kotlin.simpleName} is not enough frame to process... nSeconds $nSeconds face.size ${faces.size}")
                currentActionStatus = LivenessActionStatus.FAILED
                currentActionStatus
            }
        } else {
//            if (this.TAG == "HoldSteady2Seconds" && (isEyesClosed(faces[faces.size - 1].face) || isEyesClosed(faces[0].face))) { // Nếu đang ở HoldSteady và 1 trong 2 mắt nhắm th return false.
//                currentActionStatus = LivenessActionStatus.FAILED
//                return currentActionStatus
//            } else {
            Helpers.printLog("${this.javaClass.kotlin.simpleName} is processing... nSeconds $nMillis face.size ${faces.size}")
            currentActionStatus = individualProcess(session, faces)
            currentActionStatus
            //            }

        }

    }

    private fun isNotEnoughFrame(): Boolean {
//        Helpers.printLog("LivenessSession ${this.javaClass.simpleName} $startTime - $nSeconds isNotEnoughFrame $isNotEnoughFrame")
        return nMillis != 0 && (System.currentTimeMillis() - startTime < nMillis)
    }

    private fun getFacesByTime(session: LivenessSession, millis: Int): List<InputFace> {
        return if (millis == 0) {
            listOf(session.getFaceList()[session.faceList.size - 1])
        } else {
            if (session.getFaceList()[session.faceList.size - 1].inputTime - session.getFaceList()[0].inputTime > millis) {
                session.faceList.removeAll { session.getFaceList()[session.faceList.size - 1].inputTime - it.inputTime >= millis }
                session.getFaceList()
            } else listOf()
        }
    }
}

class HoldSteady2Seconds(seconds: Int?) : LivenessAction() {
    private var skipEyeClosedTimes = 0

    init {
        TAG = "HoldSteady2Seconds"
        if (seconds != null) nMillis = seconds
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession HoldSteady2Seconds is processing nSeconds $nSeconds ${faces.size}")
        val face1 = faces[0]
        val face2 = faces[faces.size - 1]
        if (isFaceLookStraight(face1.face) && isFaceLookStraight(face2.face)) {
            if (isEyesClosed(face2.face) && skipEyeClosedTimes < 3) {
                skipEyeClosedTimes++
                Helpers.printLog("HoldSteady2Seconds skip eye closed $skipEyeClosedTimes time(s)")
            } else
                return LivenessActionStatus.SUCCESS
        }
//        else Helpers.printLog("LivenessSession HoldSteady2Seconds not straight!")
        return LivenessActionStatus.FAILED
    }
}

class ShakeHead : LivenessAction() {
    init {
        TAG = "ShakeHead"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession ${this.javaClass.simpleName} is processing")
        return LivenessActionStatus.FAILED
    }
}

class TurnLeft : LivenessAction() {

    init {
        nMillis = 0
        TAG = "TurnLeft"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession ${this.javaClass.simpleName} is processing ${faces.size}")
        val face1 = faces[0]
        if (getFacePosition(face1.face) == LivenessFacePosition.LEFT) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class TurnRight : LivenessAction() {

    init {
        nMillis = 0
        TAG = "TurnRight"

    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
        val face1 = faces[faces.size - 1]
//        val face2 = faces[faces.size - 1]
        val facePosition = getFacePosition(face1.face)
        if (facePosition == LivenessFacePosition.RIGHT) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class TurnDown : LivenessAction() {

    init {
        nMillis = 0
        TAG = "TurnDown"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession $TAG is processing ${faces.size}")
        val face1 = faces[0]
//        val face2 = faces[faces.size - 1]
//        val facePosition = getFacePosition(face1.face, face2.face)
        val facePosition = getFacePosition(face1.face)
//        Helpers.printLog("getFacePosition Face1: $facePosition")
        if (facePosition == LivenessFacePosition.DOWN) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class TurnUp : LivenessAction() {
    init {
        nMillis = 0
        TAG = "TurnUp"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession $TAG is processing ${faces.size}")
        val face1 = faces[0]
//        val face2 = faces[faces.size - 1]
//        val facePosition = getFacePosition(face1.face, face2.face)
        val facePosition = getFacePosition(face1.face)
        Helpers.printLog("getFacePosition Face1: $facePosition")
        if (facePosition == LivenessFacePosition.UP) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class TiltRight : LivenessAction() {

    init {
        nMillis = 0
        TAG = "TiltRight"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession $TAG is processing ${faces.size}")
        val face1 = faces[0]
//        val face2 = faces[faces.size - 1]
//        val facePosition = getFacePosition(face1.face, face2.face)
        val facePosition = getFacePosition(face1.face)
        Helpers.printLog("getFacePosition Face1: $facePosition")
        if (facePosition == LivenessFacePosition.TILT_RIGHT) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class TiltLeft : LivenessAction() {

    init {
        nMillis = 0
        TAG = "TiltLeft"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
//        Helpers.printLog("LivenessSession $TAG is processing ${faces.size}")
        val face1 = faces[0]
//        val face2 = faces[faces.size - 1]
//        val facePosition = getFacePosition(face1.face, face2.face)
        val facePosition = getFacePosition(face1.face)
        Helpers.printLog("getFacePosition Face1: $facePosition")
        if (facePosition == LivenessFacePosition.TILT_LEFT) {
            return LivenessActionStatus.SUCCESS
        }
        return LivenessActionStatus.FAILED
    }
}

class ComeClose : LivenessAction() {
    init {
        TAG = "ComeClose"
        nMillis = 2000 // Increased time to allow for stable detection
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
        if (isFacesReadyToProcess(faces)) {
            val isStable = checkStability(recentFaceSizes)
            val isCloseEnough = checkProximity(recentFaceSizes.average(), FACE_AREA_BIG_THRESHOLD, FACE_AREA_TOO_BIG_THRESHOLD)
            if (isStable && isCloseEnough) {
                stableFrameCount++
                if (stableFrameCount >= REQUIRED_STABLE_FRAME_COUNT) {
                    Helpers.printLog("$TAG: Face is stable and close enough")
                    return LivenessActionStatus.SUCCESS
                }
            } else {
                Helpers.printLog("$TAG individualProcess: isStable: $isStable  isCloseEnough: $isCloseEnough - ${recentFaceSizes.average()} - $FACE_AREA_BIG_THRESHOLD - $FACE_AREA_TOO_BIG_THRESHOLD")
                stableFrameCount = 0
            }
        }
        return LivenessActionStatus.FAILED
    }

}

class GoFar : LivenessAction() {
    init {
        TAG = "GoFar"
        nMillis = 1000
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
        if (isFacesReadyToProcess(faces)) {
            val isStable = checkStability(recentFaceSizes)
            val isFarEnough = checkProximity(recentFaceSizes.average(), FACE_AREA_TOO_SMALL_THRESHOLD, FACE_AREA_SMALL_ENOUGH_THRESHOLD)
            if (isStable && isFarEnough) {
                stableFrameCount++
                if (stableFrameCount >= REQUIRED_STABLE_FRAME_COUNT) {
                    return LivenessActionStatus.SUCCESS
                }
            } else {
                Helpers.printLog("$TAG individualProcess: isStable: $isStable  isFarEnough: $isFarEnough - ${recentFaceSizes.average()} - $FACE_AREA_TOO_SMALL_THRESHOLD - $FACE_AREA_SMALL_ENOUGH_THRESHOLD")
                stableFrameCount = 0
            }
        }
        return LivenessActionStatus.FAILED
    }

    private fun calculateNormalizedFaceSize(inputFace: InputFace): Double {
        val bounds = inputFace.face.boundingBox
        val faceArea = bounds.width() * bounds.height()
        val frameArea = inputFace.frameWidth * inputFace.frameHeight
        return faceArea.toDouble() / frameArea
    }

}

open class BreakAction : LivenessAction() {
    init {
        isBreakAction = true
        TAG = "BreakAction"
    }

    override fun individualProcess(
        session: LivenessSession,
        faces: List<InputFace>,
    ): LivenessActionStatus {
        Helpers.printLog("LivenessSession ${this.javaClass.simpleName} on default process SUCCESS")
        return LivenessActionStatus.SUCCESS
    }

}

class Success : BreakAction() {
    init {
        TAG = "Success"
    }
}

class NotFollow : BreakAction() {   init {
    TAG = "NotFollow"
}
}

class Processing : BreakAction() {   init {
    TAG = "Processing"
}
}

class Timeout : BreakAction() {   init {
    TAG = "Timeout"
}
}