package vn.kalapa.ekyc.views

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.ValueAnimator
import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import androidx.interpolator.view.animation.FastOutSlowInInterpolator
import vn.kalapa.ekyc.utils.Common

class KalapaLivenessProgressView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : View(
    context, attrs, defStyleAttr
) {
    private val paint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        style = Paint.Style.STROKE
        strokeWidth = 24f
        strokeCap = Paint.Cap.ROUND
        strokeJoin = Paint.Join.ROUND
    }
    private var progressAnimator: ValueAnimator? = null
    private var currentProgress = 0f
    private val rectF = RectF()
    private var currentPart = 1
    private var livenessVersion = Common.LIVENESS_VERSION.PASSIVE
    private var totalParts = 1
    private var completedProgress = mutableMapOf<Int, Float>()

    private var successfulColor = Color.GREEN
    private var failureColor = Color.RED
    private var progressColor = successfulColor

    fun setSuccessfulColor(color: String) {
        try {
            successfulColor = Color.parseColor(color)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    fun setFailureColor(color: String) {
        try {
            failureColor = Color.parseColor(color)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private val backgroundColor = Color.LTGRAY
    private var isLastStepInProgress = false

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        val padding = paint.strokeWidth / 2
        rectF.set(
            padding,
            padding,
            w - padding,
            h - padding
        )
    }

    @SuppressLint("DrawAllocation")
    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        val centerX = width / 2f
        val centerY = height / 2f
        val radius =
            (width.coerceAtLeast(height) / 2) - paint.strokeWidth / 2

        // Draw background circle
        paint.color = backgroundColor
        canvas.drawCircle(centerX, centerY, radius, paint)

        // Draw progress
        paint.color = progressColor
        val degreesPerPart = 360f / totalParts

        // Draw completed parts
        for (part in 1 until currentPart) {
            val startAngle = -90f + (part - 1) * degreesPerPart
            canvas.drawArc(rectF, startAngle, degreesPerPart, false, paint)
        }

        // Draw current progress
        val startAngle = -90f + (currentPart - 1) * degreesPerPart
        val currentSweepAngle = currentProgress * degreesPerPart
        canvas.drawArc(rectF, startAngle, currentSweepAngle, false, paint)
    }


    private fun startAnim(duration: Long = 2000, onAnimationEnd: (() -> Unit)? = null) {
        progressAnimator?.cancel()

        progressAnimator = ValueAnimator.ofFloat(0f, 1f).apply {
            this.duration = duration
            interpolator = FastOutSlowInInterpolator()
            addUpdateListener { animator ->
                currentProgress = animator.animatedValue as Float
                invalidate()
            }
            addListener(object : AnimatorListenerAdapter() {
                override fun onAnimationEnd(animation: Animator) {
                    completedProgress[currentPart] = 1f
                    onAnimationEnd?.invoke()
                }
            })
            start()
        }
    }

    fun nextPart(duration: Long = 600) {
        if (currentPart <= totalParts) {
            if (currentPart == totalParts && isLastStepInProgress) {
                return
            }

            progressColor = successfulColor

            if (currentPart == totalParts) {
                isLastStepInProgress = true
            }

            startAnim(duration) {
                if (currentPart < totalParts) {
                    currentPart++
                    currentProgress = 0f
                }
                if (currentPart == totalParts) {
                    isLastStepInProgress = false
                }
            }
        }
    }

    fun setLivenessVersion(version: Common.LIVENESS_VERSION) {
        livenessVersion = version
        totalParts = when (version) {
            Common.LIVENESS_VERSION.PASSIVE -> 1
            Common.LIVENESS_VERSION.SEMI_ACTIVE -> 2
            Common.LIVENESS_VERSION.ACTIVE -> 3
        }
        currentPart = 1
        completedProgress.clear()
        stopProgress()
    }

    fun progressFail(duration: Long = 2000) {
        progressColor = failureColor
        currentPart = 0
        totalParts = 1
        completedProgress.clear()
        isLastStepInProgress = false
        startAnim(duration)
    }

    fun stopProgress() {
        progressAnimator?.cancel()
        currentProgress = 0f
        invalidate()
    }

    fun resetProgress() {
        progressAnimator?.cancel()
        currentProgress = 0f
        currentPart = 1
        progressColor = successfulColor
        isLastStepInProgress = false
        setLivenessVersion(livenessVersion)
        completedProgress.clear()
        isLastStepInProgress = false
        invalidate()
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        progressAnimator?.cancel()
    }
}