package vn.kalapa.ekyc.codescanner

import android.graphics.Bitmap
import android.util.Log
import com.google.mlkit.vision.barcode.BarcodeScanner
import com.google.mlkit.vision.barcode.BarcodeScannerOptions
import com.google.mlkit.vision.barcode.BarcodeScanning
import com.google.mlkit.vision.barcode.ZoomSuggestionOptions
import com.google.mlkit.vision.barcode.common.Barcode
import com.google.mlkit.vision.common.InputImage
import kotlinx.coroutines.suspendCancellableCoroutine
import java.util.concurrent.Executors
import kotlin.coroutines.resume

internal object ImageBarCodeProcessor {
    private val barCodeScannerClient: BarcodeScanner by lazy {
        val optionsBuilder = BarcodeScannerOptions
            .Builder()
            .setExecutor(
                Executors.newSingleThreadExecutor()
            )
            .setBarcodeFormats(Barcode.FORMAT_ALL_FORMATS)
            .setZoomSuggestionOptions(
                ZoomSuggestionOptions.Builder { true }.setMaxSupportedZoomRatio(8f).build()
            )
        BarcodeScanning.getClient(optionsBuilder.build())
    }

    suspend fun observerImageBarCodeFromBitmap(bitmap: Bitmap): Result<Barcode> =
        suspendCancellableCoroutine { cont ->
            val imageInput = InputImage.fromBitmap(bitmap, 0)
            barCodeScannerClient.process(imageInput)
                .addOnSuccessListener { codes ->
                    val barcode = codes.firstNotNullOfOrNull { it }
                    if (barcode != null) {
                        Log.d(
                            "ImageBarCodeProcessor",
                            "observerImageBarCodeFromBitmap: ${barcode.rawValue}"
                        )
                        cont.resume(Result.success(barcode))
                    } else {
                        cont.resume(Result.failure(NoSuchElementException("No barcode found")))
                    }
                }
                .addOnFailureListener { exception ->
                    Log.e("ImageBarCodeProcessor", "observerImageBarCodeFromBitmap: $exception")
                    cont.resume(Result.failure(exception))
                }
        }

}