package vn.kalapa.ekyc.iproov.component

import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import android.view.animation.DecelerateInterpolator
import vn.kalapa.ekyc.KalapaSDK
import kotlin.math.min

class CircularProgressView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : View(context, attrs, defStyleAttr) {

    private val progressColor = Color.parseColor(KalapaSDK.config.mainColor)
    private val backgroundColor = Color.parseColor("#E0E0E0")
    private val progressWidth = 20f
    private val textColor = Color.BLACK
    private val textSize = 40f

    private val backgroundPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        style = Paint.Style.STROKE
        strokeCap = Paint.Cap.ROUND
        color = backgroundColor
        strokeWidth = progressWidth
    }

    private val progressPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        style = Paint.Style.STROKE
        strokeCap = Paint.Cap.ROUND
        color = progressColor
        strokeWidth = progressWidth
    }

    private val textPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        textAlign = Paint.Align.CENTER
        color = textColor
        textSize = this@CircularProgressView.textSize
    }

    private var animatedProgress: Double = 0.0

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        val diameter = min(width, height).toFloat()
        val centerX = width / 2f
        val centerY = height / 2f
        val radius = diameter / 2f - progressWidth / 2f

        canvas.drawCircle(centerX, centerY, radius, backgroundPaint)

        val progressRect = RectF(
            centerX - radius,
            centerY - radius,
            centerX + radius,
            centerY + radius
        )
        canvas.drawArc(
            progressRect,
            -90f,
            (animatedProgress * 360).toFloat(),
            false,
            progressPaint
        )

        val progressText = "${(animatedProgress * 100).toInt()}%"
        canvas.drawText(
            progressText,
            centerX,
            centerY - (textPaint.descent() + textPaint.ascent()) / 2f,
            textPaint
        )
    }

    fun setProgressAnimated(targetProgress: Double, duration: Long = 300) {
        val safeProgress = targetProgress.coerceIn(0.0, 1.0)

        ValueAnimator.ofFloat(animatedProgress.toFloat(), safeProgress.toFloat()).apply {
            this.duration = duration
            interpolator = DecelerateInterpolator()
            addUpdateListener { animator ->
                animatedProgress = (animator.animatedValue as Float).toDouble()
                invalidate()
            }
            start()
        }
    }
}