package vn.kalapa.ekyc.managers

import android.content.Context
import com.google.gson.Gson
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import vn.kalapa.ekyc.models.KalapaAllLanguageModel
import vn.kalapa.ekyc.models.KalapaLanguageModel
import vn.kalapa.ekyc.utils.Helpers
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream

object KLPCDNManager {

    private const val FILE_LANGUAGE = "language.json"
    private const val KEY_LANGUAGE_VERSION = "klp_version_language"
    private const val KEY_CARD_MODEL_VERSION = "klp_version_card_model"
    private const val FILE_CARD_MODEL = "card_model.tflite"

    fun saveJsonStringIntoFile(context: Context, klpLanguageModel: KalapaLanguageModel) {
        val jsonString = Gson().toJson(klpLanguageModel)
        val version = if (klpLanguageModel.data?.version_id != null) klpLanguageModel.data.version_id else ""
        val file = File(context.filesDir, FILE_LANGUAGE.replace(".json", "${if (version.isNotEmpty()) "_$version" else ""}.json"))
        file.writeText(jsonString, Charsets.UTF_8)
        version.isNotEmpty().let {
            Helpers.savePrefs(KEY_LANGUAGE_VERSION, version)
        }
    }

    suspend fun saveModelIntoFile(context: Context, version: String? = "", inputStream: InputStream) {
        withContext(Dispatchers.IO) {
            val startTime = System.currentTimeMillis()
            val file = File(context.filesDir, FILE_CARD_MODEL.replace(".tflite", "$version.tflite"))
            try {
                FileOutputStream(file).use { output ->
                    inputStream.copyTo(output)
                }
                Helpers.printLog("KLPCardModelManager Saved to cache ${file.name}: ${file.length() * 1f / (1024 * 1024)} MB in ${System.currentTimeMillis() - startTime}")
                version?.let { Helpers.savePrefs(KEY_CARD_MODEL_VERSION, it) }
                Thread.sleep(200)
            } catch (e: IOException) {
                e.printStackTrace()
            }
        }
    }

    fun loadModelFromCache(context: Context, version: String?): File? {
        version?.let {
            val cachedVersion = Helpers.getStringPreferences(KEY_CARD_MODEL_VERSION)
            Helpers.printLog("loadModelFromCache cachedVersion $cachedVersion - $version")
            if (cachedVersion == version) {
                return isCachedFileExists(context, FILE_CARD_MODEL.replace(".tflite", "$version.tflite"))
            }
        }
        return null

    }

    fun loadLanguageFromCache(context: Context, version: String?): KalapaLanguageModel? {
        version?.let {
            val cachedVersion = Helpers.getStringPreferences(KEY_LANGUAGE_VERSION)
            Helpers.printLog("loadLanguageFromCache cachedVersion $cachedVersion - $version")
            if (cachedVersion == version) {
                isCachedFileExists(context, FILE_LANGUAGE.replace(".json", "${if (version.isNotEmpty()) "_$version" else ""}.json"))?.let {
                    return try {
                        val content = it.readText(Charsets.UTF_8)
                        Helpers.printLog("Load language from cache $content")
                        KalapaLanguageModel.fromJson(content)
                    } catch (e: Exception) {
                        Helpers.printLog("Cache file not found ${e.localizedMessage}")
                        null
                    }
                }
            }
        }
        return null
    }


    private fun isCachedFileExists(context: Context, fileName: String): File? {
        val file = File(context.filesDir, fileName)
        Helpers.printLog("File: $fileName exists? ${file.exists()}")
        return if (file.exists()) file else null
    }

}