package vn.kalapa.ekyc.codescanner

import android.graphics.Bitmap
import androidx.appcompat.widget.ThemedSpinnerAdapter.Helper
import androidx.camera.core.ExperimentalGetImage
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageProxy
import com.google.mlkit.vision.barcode.BarcodeScanner
import com.google.mlkit.vision.barcode.BarcodeScannerOptions
import com.google.mlkit.vision.barcode.BarcodeScanning
import com.google.mlkit.vision.barcode.ZoomSuggestionOptions
import com.google.mlkit.vision.barcode.common.Barcode
import com.google.mlkit.vision.common.InputImage
import vn.kalapa.ekyc.utils.Helpers

internal class QRCodeAnalyzer(
    private val onSuccess: (barCode: Barcode, imageRaw: Bitmap) -> Unit,
    private val onFailure: ((Exception) -> Unit),
    private val onPassCompleted: ((Boolean) -> Unit),
    private val suggestZoomRatio: (Float) -> Unit = {},
) : ImageAnalysis.Analyzer {

    // catch if for some reason MlKitContext has not been initialized
    private val barcodeScanner: BarcodeScanner? by lazy {

        val optionsBuilder = BarcodeScannerOptions
            .Builder()
            .setZoomSuggestionOptions(ZoomSuggestionOptions.Builder { ratio ->
                suggestZoomRatio(ratio)
                true
            }.build())
            .setBarcodeFormats(Barcode.FORMAT_QR_CODE)
        try {
            BarcodeScanning.getClient(optionsBuilder.build())
        } catch (e: Exception) {
            onFailure(e)
            null
        }
    }

    // Remove cached state from CPU capture
    @Volatile
    private var failureOccurred = false
    private var failureTimestamp = 0L

    @Volatile
    private var qrCodeDetected = false

    @ExperimentalGetImage
    override fun analyze(imageProxy: ImageProxy) {
        if (imageProxy.image == null || qrCodeDetected) {
            imageProxy.close()
            return
        }

        if (failureOccurred && System.currentTimeMillis() - failureTimestamp < 1000L) {
            imageProxy.close()
            return
        }

        failureOccurred = false

        val bitmap = BitmapUtils.getBitmap(imageProxy)
        if (bitmap == null) {
            imageProxy.close()
            return
        }

        val image = InputImage.fromBitmap(bitmap, 0)

        barcodeScanner?.let { scanner ->
            if (!qrCodeDetected)
                scanner.process(image)
                    .addOnSuccessListener { codes ->
                        codes.firstNotNullOfOrNull { it }?.let {
                            qrCodeDetected = true
                            onSuccess(it, bitmap)
                        }
                    }
                    .addOnFailureListener {
                        failureOccurred = true
                        failureTimestamp = System.currentTimeMillis()
                        onFailure(it)
                    }
                    .addOnCompleteListener {
                        onPassCompleted(failureOccurred)
                        imageProxy.close()
                    }
        }
    }

    fun reset() {
        qrCodeDetected = false
    }
}