package vn.kalapa.ekyc.codescanner

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Path
import android.graphics.Rect
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import android.view.animation.AccelerateDecelerateInterpolator

class QrDetectionView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : View(context, attrs, defStyleAttr) {

    private var boundingBox: RectF? = null
    private var targetBoundingBox: RectF? = null

    private val cornerPaint = Paint().apply {
        style = Paint.Style.STROKE
        strokeWidth = 12f
    }

    private val cornerLength = 40f

    private val startColor = Color.YELLOW
    private val endColor = Color.GREEN

    private var currentColor = startColor

    private var scaleProgress = 0f
    private var colorProgress = 0f
    private val initialScale = 1.5f
    private val path = Path()


    private val propertyAnimator = ValueAnimator.ofFloat(0f, 1f).apply {
        duration = 300L
        interpolator = AccelerateDecelerateInterpolator()
        addUpdateListener { animation ->
            scaleProgress = animation.animatedValue as Float
            colorProgress = animation.animatedValue as Float
            updateColor()
            invalidate()
        }
    }

    private fun updateColor() {
        val red = evaluateColor(Color.red(startColor), Color.red(endColor), colorProgress)
        val green = evaluateColor(Color.green(startColor), Color.green(endColor), colorProgress)
        val blue = evaluateColor(Color.blue(startColor), Color.blue(endColor), colorProgress)
        currentColor = Color.rgb(red, green, blue)
        cornerPaint.color = currentColor
    }

    private fun evaluateColor(start: Int, end: Int, fraction: Float): Int {
        return (start + (end - start) * fraction).toInt()
    }

    fun showDetection(
        rect: Rect,
        previewWidth: Int,
        previewHeight: Int,
        onAnimFinished: () -> Unit,
    ) {
        val scaleX = width.toFloat() / previewWidth
        val scaleY = height.toFloat() / previewHeight

        val scaledLeft = rect.left * scaleX
        val scaledTop = rect.top * scaleY
        val scaledBottom = rect.bottom * scaleY

        val boxHeight = scaledBottom - scaledTop
        val centerX = scaledLeft + (rect.width() * scaleX) / 2

        targetBoundingBox = RectF(
            centerX - boxHeight / 2,
            scaledTop,
            centerX + boxHeight / 2,
            scaledBottom
        )

        val centerY = targetBoundingBox!!.centerY()
        val targetSize = boxHeight // Use height as the size reference

        boundingBox = RectF(
            centerX - targetSize * initialScale / 2,
            centerY - targetSize * initialScale / 2,
            centerX + targetSize * initialScale / 2,
            centerY + targetSize * initialScale / 2
        )

        propertyAnimator.cancel()
        propertyAnimator.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                onAnimFinished()
            }
        })
        propertyAnimator.start()
    }

    fun hideDetection() {
        propertyAnimator.cancel()
        boundingBox = null
        targetBoundingBox = null
        invalidate()
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        boundingBox?.let { initialBox ->
            targetBoundingBox?.let { targetBox ->
                // Interpolate between initial and target positions
                val currentLeft = lerp(initialBox.left, targetBox.left, scaleProgress)
                val currentTop = lerp(initialBox.top, targetBox.top, scaleProgress)
                val currentRight = lerp(initialBox.right, targetBox.right, scaleProgress)
                val currentBottom = lerp(initialBox.bottom, targetBox.bottom, scaleProgress)

                val currentBox = RectF(currentLeft, currentTop, currentRight, currentBottom)

                path.reset()

                // Top-left corner
                path.moveTo(currentBox.left + cornerLength, currentBox.top)
                path.lineTo(currentBox.left, currentBox.top)
                path.lineTo(currentBox.left, currentBox.top + cornerLength)

                // Top-right corner
                path.moveTo(currentBox.right - cornerLength, currentBox.top)
                path.lineTo(currentBox.right, currentBox.top)
                path.lineTo(currentBox.right, currentBox.top + cornerLength)

                // Bottom-left corner
                path.moveTo(currentBox.left, currentBox.bottom - cornerLength)
                path.lineTo(currentBox.left, currentBox.bottom)
                path.lineTo(currentBox.left + cornerLength, currentBox.bottom)

                // Bottom-right corner
                path.moveTo(currentBox.right - cornerLength, currentBox.bottom)
                path.lineTo(currentBox.right, currentBox.bottom)
                path.lineTo(currentBox.right, currentBox.bottom - cornerLength)

                // Draw the corners
                canvas.drawPath(path, cornerPaint)
            }
        }
    }

    private fun lerp(start: Float, end: Float, fraction: Float): Float {
        return start + (end - start) * fraction
    }
}