package vn.kalapa.ekyc.codescanner

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.BlurMaskFilter
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Path
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.RectF
import android.text.Layout
import android.text.StaticLayout
import android.text.TextPaint
import android.util.AttributeSet
import android.view.View
import vn.kalapa.R
import vn.kalapa.ekyc.managers.KLPLanguageManager

class QRScannerOverlayView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : View(context, attrs, defStyleAttr) {

    private val framePaint = Paint().apply {
        color = Color.WHITE
        style = Paint.Style.STROKE
        strokeWidth = dpToPx(4f)
    }

    private val textPaint = Paint().apply {
        color = Color.WHITE
        textSize = dpToPx(16f)
        isAntiAlias = true
    }

    private val overlayPaint = Paint().apply {
        color = Color.parseColor("#BF000000")
    }

    private val cornerLength = dpToPx(25f)
    private val frameRect = RectF()
    private val scannerFrameSize = 250f
    private val blurMaskFilter = BlurMaskFilter(20f, BlurMaskFilter.Blur.NORMAL)

    init {
        overlayPaint.maskFilter = blurMaskFilter
        setLayerType(LAYER_TYPE_HARDWARE, null)
    }

    @SuppressLint("DrawAllocation")
    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        canvas.drawRect(0f, 0f, width.toFloat(), height.toFloat(), overlayPaint)

        val frameWidth = dpToPx(scannerFrameSize)
        val frameHeight = dpToPx(scannerFrameSize)
        val left = (width - frameWidth) / 2f
        val top = (height - frameHeight) / 2f
        frameRect.set(left, top, left + frameWidth, top + frameHeight)

        canvas.drawRect(frameRect, Paint().apply {
            style = Paint.Style.FILL
            xfermode = PorterDuffXfermode(PorterDuff.Mode.CLEAR)
        })

        // Top-left corner
        canvas.drawPath(Path().apply {
            moveTo(frameRect.left, frameRect.top + cornerLength)
            lineTo(frameRect.left, frameRect.top)
            lineTo(frameRect.left + cornerLength, frameRect.top)
        }, framePaint)

        // Top-right corner
        canvas.drawPath(Path().apply {
            moveTo(frameRect.right - cornerLength, frameRect.top)
            lineTo(frameRect.right, frameRect.top)
            lineTo(frameRect.right, frameRect.top + cornerLength)
        }, framePaint)

        // Bottom-left corner
        canvas.drawPath(Path().apply {
            moveTo(frameRect.left, frameRect.bottom - cornerLength)
            lineTo(frameRect.left, frameRect.bottom)
            lineTo(frameRect.left + cornerLength, frameRect.bottom)
        }, framePaint)

        // Bottom-right corner
        canvas.drawPath(Path().apply {
            moveTo(frameRect.right - cornerLength, frameRect.bottom)
            lineTo(frameRect.right, frameRect.bottom)
            lineTo(frameRect.right, frameRect.bottom - cornerLength)
        }, framePaint)

        val text = "" // KLPLanguageManager.get(context.getString(R.string.klp_qr_code_scan_guide))
        val padding = dpToPx(16f)
        val textWidth = width - 2 * padding


        val staticLayout = StaticLayout.Builder.obtain(
            text,
            0,
            text.length,
            TextPaint(textPaint),
            textWidth.toInt()
        )
            .setAlignment(Layout.Alignment.ALIGN_CENTER)
            .setLineSpacing(0f, 1f)
            .setIncludePad(false)
            .build()

        val textY = frameRect.bottom + dpToPx(16f)
        val tapTextY = frameRect.top - staticLayout.height - dpToPx(16f)
        canvas.save()
        canvas.translate(padding, textY)
        staticLayout.draw(canvas)
        canvas.restore()

        val aboveText = KLPLanguageManager.get(context.getString(R.string.klp_tap_to_focus))
        val aboveStaticLayout = StaticLayout.Builder.obtain(
            aboveText,
            0,
            aboveText.length,
            TextPaint(textPaint),
            textWidth.toInt()
        )
            .setAlignment(Layout.Alignment.ALIGN_CENTER)
            .setLineSpacing(0f, 1f)
            .setIncludePad(false)
            .build()
        canvas.save()
        canvas.translate(padding, tapTextY)
        aboveStaticLayout.draw(canvas)
        canvas.restore()
    }

    private fun dpToPx(dp: Float): Float {
        return dp * context.resources.displayMetrics.density
    }
}