package vn.kalapa.ekyc.iproov

import android.content.Context
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONObject
import vn.kalapa.ekyc.KalapaSDK
import vn.kalapa.ekyc.KalapaStrictLivenessHandler
import vn.kalapa.ekyc.models.IPRAccessToken
import vn.kalapa.ekyc.models.KalapaError
import vn.kalapa.ekyc.networks.Client
import vn.kalapa.ekyc.networks.KalapaAPI
import vn.kalapa.ekyc.utils.EventChannel
import vn.kalapa.ekyc.utils.HasEventFlow
import vn.kalapa.ekyc.utils.Helpers
import vn.kalapa.ekyc.utils.IProovConstant

sealed interface IProovFaceVerificationState {
    object Loading : IProovFaceVerificationState
    object TokenSuccess : IProovFaceVerificationState
    object StartScanning : IProovFaceVerificationState
    object Expired : IProovFaceVerificationState
    data class LivenessSuccess(val base64Image: String) : IProovFaceVerificationState
    data class Error(val message: String) : IProovFaceVerificationState
}

class IProovFaceVerificationViewModel(
    private val eventChannel: EventChannel<IProovFaceVerificationState> = EventChannel(),
) : ViewModel(), HasEventFlow<IProovFaceVerificationState> by eventChannel {
    private var userID = KalapaSDK.session + System.currentTimeMillis()
    var token = ""
    private var job: Job? = null

    fun startIProov(context: Context) {
        userID = KalapaSDK.session + System.currentTimeMillis()
        job?.cancel()

        job = viewModelScope.launch(Dispatchers.IO) {
            try {
//                busEvent(IProovFaceVerificationState.Loading)
                if (KalapaSDK.session == "") {
                    busEvent(IProovFaceVerificationState.StartScanning) // For custom test
                } else
                    KalapaAPI.getStrictLivenessToken("onboarding",
                        object : Client.RequestListener {
                            override fun success(jsonObject: JSONObject) {
                                val accessToken = IPRAccessToken.fromJson(jsonObject.toString())
                                Helpers.printLog("getStrictLivenessToken $accessToken")
                                if (accessToken.data != null && accessToken.data.access_token.isNotEmpty()) {
                                    token = accessToken.data.access_token
                                    Helpers.printLog("getStrictLivenessToken token: $token")
                                    busEvent(IProovFaceVerificationState.StartScanning)
                                }
                            }

                            override fun fail(error: KalapaError) {
                                busEvent(IProovFaceVerificationState.Error(error.getMessageError()))
                            }

                            override fun timeout() {
                                busEvent(IProovFaceVerificationState.Expired)
                            }
                        }
                    )
            } catch (ex: Exception) {
            }
        }
    }


    fun getResult(client: String, activity: IProovFaceVerificationActivity) {
        viewModelScope.launch {
            busEvent(IProovFaceVerificationState.Loading)
            (KalapaSDK.handler as KalapaStrictLivenessHandler).process(token, activity)

//            val result = IProovApi.validateClaimToken(token, userID, client)
//            result.onSuccess { response ->
//                busEvent(IProovFaceVerificationState.LivenessSuccess(response.frame))
//            }
//            result.onFailure { error ->
//                busEvent(
//                    IProovFaceVerificationState.Error(
//                        error.message ?: "Error validating claim"
//                    )
//                )
//            }
        }
    }

    private fun busEvent(event: IProovFaceVerificationState) {
        viewModelScope.launch {
            eventChannel.send(event)
        }
    }

    override fun onCleared() {
        super.onCleared()
        job?.cancel()
    }
}