package vn.kalapa.ekyc.iproov

import android.app.Dialog
import android.content.res.ColorStateList
import android.graphics.Color
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.widget.ProgressBar
import android.widget.TextView
import android.widget.Toast
import androidx.activity.viewModels
import androidx.annotation.DrawableRes
import androidx.appcompat.widget.AppCompatButton
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.view.children
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import com.iproov.sdk.api.IProov
import com.iproov.sdk.api.exception.CameraException
import com.iproov.sdk.api.exception.CaptureAlreadyActiveException
import com.iproov.sdk.api.exception.ConsentRequiredException
import com.iproov.sdk.api.exception.FaceDetectorException
import com.iproov.sdk.api.exception.IProovException
import com.iproov.sdk.api.exception.InvalidOptionsException
import com.iproov.sdk.api.exception.MultiWindowUnsupportedException
import com.iproov.sdk.api.exception.NetworkException
import com.iproov.sdk.api.exception.ServerException
import com.iproov.sdk.api.exception.SessionCannotBeStartedTwiceException
import com.iproov.sdk.api.exception.UnexpectedErrorException
import com.iproov.sdk.api.exception.UnsupportedDeviceException
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import vn.kalapa.R
import vn.kalapa.databinding.ActivityIproovFaceVerificationBinding
import vn.kalapa.ekyc.DialogListener
import vn.kalapa.ekyc.KalapaCaptureHandler
import vn.kalapa.ekyc.KalapaSDK
import vn.kalapa.ekyc.KalapaSDKMediaType
import vn.kalapa.ekyc.KalapaSDKResultCode
import vn.kalapa.ekyc.KalapaStrictLivenessHandler
import vn.kalapa.ekyc.activity.BaseActivity
import vn.kalapa.ekyc.managers.KLPLanguageManager
import vn.kalapa.ekyc.utils.Helpers
import vn.kalapa.ekyc.utils.IProovConstant
import vn.kalapa.ekyc.utils.KALAPA_LOG_ACTION
import vn.kalapa.ekyc.utils.KALAPA_LOG_LEVEL
import vn.kalapa.ekyc.utils.LogUtil.logEndSession
import vn.kalapa.ekyc.utils.LogUtil.logEvent
import vn.kalapa.ekyc.views.ProgressView
import java.util.Locale

class IProovFaceVerificationActivity : BaseActivity() {
    private lateinit var binding: ActivityIproovFaceVerificationBinding
    private var sessionStateJob: Job? = null
    private var failureCounter = 0
    private val viewModel: IProovFaceVerificationViewModel by viewModels()
    private val loadingDialog by lazy { createLoadingDialog() }

    private fun createLoadingDialog(): Dialog = Dialog(this).apply {
        setContentView(R.layout.kalapa_progress_view)
        findViewById<ProgressBar>(R.id.progress_container_progressbar).indeterminateTintList =
            ColorStateList.valueOf(Color.parseColor(KalapaSDK.config.mainColor))
        findViewById<TextView>(R.id.custom_dialog_body).text =
            KLPLanguageManager.get(getString(R.string.klp_please_wait))
        setCancelable(false)
        window?.setBackgroundDrawableResource(android.R.color.transparent)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        Helpers.printLog("${this@IProovFaceVerificationActivity::class.java} - Locale: ${Locale.getDefault()}")
        binding = ActivityIproovFaceVerificationBinding.inflate(layoutInflater)
        setContentView(binding.root)
        setupViews()
        setupActions()
        observeStates()
    }

    private fun setupViews() {
        with(binding.btnIProovStart) {
            setTextColor(Color.parseColor(KalapaSDK.config.btnTextColor))
            backgroundTintList =
                ColorStateList.valueOf(Color.parseColor(KalapaSDK.config.mainColor))
        }
        processTextTranslated()
    }

    private fun setupActions() {
        binding.ivCloseIproov.setOnClickListener { handleBackPressed() }
        binding.btnIProovStart.setOnClickListener {
            loadingDialog.show()
            viewModel.startIProov(this)
        }
    }

    private fun observeStates() {
        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.eventFlow.collect(::handleEvents)
            }
        }
    }

    private fun handleEvents(event: IProovFaceVerificationState) {
        when (event) {
            is IProovFaceVerificationState.Error -> showErrorDialog("liveness_verification_failed", event.message)
            is IProovFaceVerificationState.LivenessSuccess -> handleLivenessSuccess(event)
            is IProovFaceVerificationState.Loading -> {
                loadingDialog.show()
                removeProgressDialogFragment()
            }

            is IProovFaceVerificationState.Expired -> sendExpired()
            is IProovFaceVerificationState.TokenSuccess -> Unit
            is IProovFaceVerificationState.StartScanning -> startScan()
        }
    }

    private fun removeProgressDialogFragment() {
        supportFragmentManager.findFragmentByTag("progressDialog")?.let {
            supportFragmentManager.beginTransaction().remove(it).commit()
        }
    }

    private fun handleLivenessSuccess(event: IProovFaceVerificationState.LivenessSuccess) {
        removeProgressDialogFragment()
        (KalapaSDK.handler as? KalapaCaptureHandler)?.process(
            base64 = event.base64Image,
            mediaType = KalapaSDKMediaType.PORTRAIT,
            callback = this@IProovFaceVerificationActivity
        )
    }

    private fun handleBackPressed() {
        Helpers.showEndKYC(this, SCREEN_ID, object : DialogListener {
            override fun onYes() {
                logEndSession(this@IProovFaceVerificationActivity, KALAPA_LOG_ACTION.EKYC_LEAVE, SCREEN_ID)
                KalapaSDK.handler.onError(KalapaSDKResultCode.USER_LEAVE)
                finish()
            }

            override fun onNo() = Unit
        })
    }

    @Throws(SessionCannotBeStartedTwiceException::class)
    private fun startScan() {
//        Toast.makeText(this@IProovFaceVerificationActivity, getString(R.string.iproov__progress_loading) + Locale.getDefault().language + getString(R.string.TEST_LANGUAGE), Toast.LENGTH_LONG).show()
        Helpers.printLog("startScan ${viewModel.token}")
        IProov.createSession(
            this,
            IProovConstant.BASE_URL,
            viewModel.token,
//            "3bbe0b8cace0852e5c3a7762a178b4b47ebe4e154354b4d18997036d1801vi10",
            IProov.Options().apply {
                title = KLPLanguageManager.get(getString(R.string.klp_liveness_title))
                titleTextColor = Color.parseColor(KalapaSDK.config.btnTextColor)
                headerBackgroundColor = Color.parseColor(KalapaSDK.config.backgroundColor)
                surroundColor = Color.parseColor(KalapaSDK.config.mainColor)
                promptTextColor = Color.parseColor(KalapaSDK.config.mainTextColor)
                promptBackgroundColor = Color.parseColor(KalapaSDK.config.backgroundColor)
            }
        ).let { session ->
            observeSessionState(session) { session.start() }
        }
    }

    private fun observeSessionState(session: IProov.Session, whenReady: (() -> Unit)? = null) {
        sessionStateJob?.cancel()
        sessionStateJob = lifecycleScope.launch(Dispatchers.IO) {
            session.state
                .onSubscription { whenReady?.invoke() }
                .collect { state ->
                    if (sessionStateJob?.isActive == true) {
                        withContext(Dispatchers.Main) {
                            handleSessionState(state)
                        }
                    }
                }
        }
    }

    private fun handleSessionState(state: IProov.State) {
        when (state) {
            is IProov.State.Starting -> loadingDialog.show()
            is IProov.State.Connecting -> Unit
            is IProov.State.Connected -> loadingDialog.dismiss()
            is IProov.State.Processing -> updateProcessingDialog(state)
            is IProov.State.Success -> {
                Helpers.printLog("handleSessionState - Success")
                viewModel.getResult(packageName, this@IProovFaceVerificationActivity)
            }

            is IProov.State.Failure -> handleFailureState(state)
            is IProov.State.Error -> {
                Helpers.printLog("Error ${state.exception}")
                showErrorDialog("liveness_verification_failed", getMessageFromIProovException(state.exception))
            }

            is IProov.State.Canceled -> showErrorDialog("liveness_verification_failed", "liveness_user_leave")
        }
    }

    private fun getMessageFromIProovException(exception: IProovException): String {
        return when (exception) {
            is NetworkException -> resources.getString(R.string.klp_error_network) // KLPLanguageManager.get(resources.getString(R.string.klp_error_network))
            is MultiWindowUnsupportedException -> resources.getString(R.string.klp_error_multi_window_mode_unsupported)
            is CaptureAlreadyActiveException -> resources.getString(R.string.klp_error_capture_already_active)
            is CameraException -> resources.getString(R.string.klp_error_create_camera)
            is FaceDetectorException -> resources.getString(R.string.klp_error_face_detector)
            is ServerException -> resources.getString(R.string.klp_error_service_temporary_down)
            is UnsupportedDeviceException -> resources.getString(R.string.klp_error_device_unsupported)
            is InvalidOptionsException -> resources.getString(R.string.klp_error_invalid_options)
            is ConsentRequiredException -> resources.getString(R.string.klp_error_consent_required)
            else -> KLPLanguageManager.get(resources.getString(R.string.klp_error_unknown))
        }
    }

    private fun updateProcessingDialog(state: IProov.State.Processing) {
        runCatching {
            val currentDialog =
                supportFragmentManager.findFragmentByTag("progressDialog") as? IProovProgressDialog
                    ?: IProovProgressDialog.newInstance().also {
                        it.show(supportFragmentManager, "progressDialog")
                    }
            currentDialog.setProgress(state.progress, state.message)
        }
    }

    private fun handleFailureState(state: IProov.State.Failure) {
        removeProgressDialogIfExists()
        Helpers.printLog("handleFailureState $state")
        if (KalapaSDK.isHandlerInitialized() && KalapaSDK.handler is KalapaStrictLivenessHandler) {
            Helpers.printLog("handleFailureState - validateOnFail")
            (KalapaSDK.handler as KalapaStrictLivenessHandler).validateOnFail(viewModel.token)
        }
        if (++failureCounter >= MAX_ATTEMPTS) {
            showErrorDialog("liveness_verification_failed", "liveness_error_too_many_attempts") { finish() }
            return
        }

        val errorMessagePair: Pair<Pair<String, String>, Int> = when (state.failureResult.reason) {
            IProov.FailureReason.TOO_MUCH_MOVEMENT -> Pair(
                Pair("failure_too_much_movement", "liveness_error_too_much_movement"),
                R.drawable.liveness_hold_still
            )

            IProov.FailureReason.TOO_BRIGHT -> Pair(
                Pair("failure_too_bright", "liveness_error_too_bright"),
                R.drawable.liveness_too_bright
            )

            IProov.FailureReason.TOO_DARK -> Pair(
                Pair("failure_too_dark", "liveness_error_too_dark"),
                R.drawable.liveness_too_dark
            )

            IProov.FailureReason.MISALIGNED_FACE -> Pair(
                Pair("failure_misaligned_face", "liveness_error_angle"),
                R.drawable.liveness_angle
            )

            IProov.FailureReason.EYES_CLOSED -> Pair(
                Pair("failure_eyes_closed", "liveness_error_eyes_closed"),
                R.drawable.liveness_eye_closed
            )

            IProov.FailureReason.FACE_TOO_FAR -> Pair(
                Pair("failure_face_too_far", "liveness_message_too_far"),
                R.drawable.liveness_too_far
            )

            IProov.FailureReason.FACE_TOO_CLOSE -> Pair(
                Pair("failure_face_too_close", "liveness_message_too_close"),
                R.drawable.liveness_too_close
            )

            IProov.FailureReason.SUNGLASSES -> Pair(
                Pair("failure_sunglasses", "liveness_message_sunglasses"),
                R.drawable.liveness_sun_glasses
            )

            IProov.FailureReason.OBSCURED_FACE -> Pair(
                Pair("failure_obscured_face", "liveness_message_obscured_face"),
                R.drawable.liveness_face_cover
            )

            IProov.FailureReason.MULTIPLE_FACES -> Pair(
                Pair("failure_multiple_faces", "liveness_message_many_faces"),
                R.drawable.liveness_multiple_faces
            )

            else -> Pair(Pair("liveness_verification_failed", "liveness_common_error"), R.drawable.liveness_fail)
        }

        showErrorDialog(
            errorMessagePair.first.first,
            errorMessagePair.first.second,
            true,
            errorMessagePair.second
        ) {
            loadingDialog.show()
            viewModel.startIProov(this)
        }
    }

    private fun showErrorDialog(
        title: String,
        message: String,
        isRetry: Boolean = false,
        @DrawableRes iconRes: Int = R.drawable.liveness_fail,
        onConfirmedClicked: (() -> Unit)? = null,
    ) {
        loadingDialog.dismiss()
        removeProgressDialogIfExists()
        Helpers.printLog("!!! showErrorDialog $message")
        Helpers.showDialog(
            this,
            KLPLanguageManager.get(title).ifEmpty { title },
            KLPLanguageManager.get(message).ifEmpty { message },
            iconRes,
            alertListener = onConfirmedClicked,
            isRetry = isRetry,
            customLayout = R.layout.custom_liveness_layout
        )
    }

    private fun removeProgressDialogIfExists() {
        supportFragmentManager.fragments.filterIsInstance<IProovProgressDialog>()
            .forEach { it.dismiss() }
    }

    private fun processTextTranslated() {
        binding.root.traverseViews { view ->
            when (view) {
                is AppCompatButton -> view.text = KLPLanguageManager.get(view.text.toString())
                is AppCompatTextView -> view.text = KLPLanguageManager.get(view.text.toString())
            }
        }
    }

    private fun View.traverseViews(action: (View) -> Unit) {
        action(this)
        if (this is ViewGroup) {
            children.forEach { it.traverseViews(action) }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        sessionStateJob?.cancel()
    }

    override fun sendError(message: String?) {
        Helpers.printLog("!!! sendError $message")
        ProgressView.hideProgress()
        showErrorDialog("liveness_verification_failed", message ?: KLPLanguageManager.get(resources.getString(R.string.klp_error_unknown)), true)
    }

    override fun sendDone(nextAction: () -> Unit) {
        logEvent(this, KALAPA_LOG_LEVEL.INFO, KALAPA_LOG_ACTION.LIVENESS_SUCCESS, SCREEN_ID)
        nextAction()
        runOnUiThread {
            ProgressView.hideProgress()
        }
        Helpers.printLog("CameraXSelfieActivity onFinish")
        finish()
    }

    companion object {
        private const val MAX_ATTEMPTS = 3
    }
}