package vn.kalapa.ekyc.liveness

import android.content.Context
import android.graphics.Bitmap
import android.os.CountDownTimer
import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import vn.kalapa.ekyc.KalapaSDK
import vn.kalapa.ekyc.managers.KLPFaceDetectorListener
import vn.kalapa.ekyc.utils.Common
import vn.kalapa.ekyc.utils.Helpers
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.TimeUnit

class LivenessHandler(
    private val context: Context,
    private val livenessSessionType: Common.LIVENESS_VERSION,
    private val faceDetectorListener: KLPFaceDetectorListener,
    private val rotationAngle: Int = 0
) {
    companion object {
        private const val PROCESS_INTERVAL = 33L // ~30 FPS processing rate
    }

    /**
     * The current additional rotation (in degrees). This is the value that will be applied
     * to the raw camera rotation. For testing the default is set to 180.
     */
    private var addedRotation: Int


    private val TAG = LivenessHandler::class.java.simpleName
    private val LIVENESS_MAX_TIME = if (KalapaSDK.isConfigInitialized() && KalapaSDK.config.livenessTimeoutInSeconds > 0) KalapaSDK.config.livenessTimeoutInSeconds * 1000L else 61_000L // 1p
    private var countdown = LIVENESS_MAX_TIME
    private var livenessSession = LivenessSession(livenessSessionType)
    private var isStop = false
    private var countDownTimer: CountDownTimer? = null
    private var startTime = System.currentTimeMillis()
    private var lastCheckTime = startTime

    private var sessionStatus: LivenessSessionStatus = LivenessSessionStatus.UNVERIFIED
    private var sessionAction: String = ""

    // Timing for orientation checking (milliseconds)
    private var TIME_PERIOD: Long = 5_000
    private val CAMERA_ORIENTATION_KEY = "KLP_CAMERA_ORIENTATION_KEY"

    private var actualRotationDegree: Int = rotationAngle

//    fun getCameraKey(): String {
//        return CAMERA_ORIENTATION_KEY
//    }

    /**
     * When true, the system is in orientation correction mode.
     */
    var checkingForOrientation = false


    // These variables track the ratios for triggering the orientation suspect callback.
    private var wrongFaceRatioCount = 0
    private var wrongFaceRatio: Double = 0.0

    // Counters for orientation suspect check (used when not in checkingForOrientation mode).
    private var sessionStatusCount = 0

    // New variables for orientation correction mode.
    private var continuousFaceCounter = 0


    private var lastProcessTime = 0L

    private val processingQueue = LinkedBlockingQueue<Pair<Bitmap, Bitmap>>(2)
    private val processingScope = CoroutineScope(Dispatchers.Default + SupervisorJob())


    init {
        // Initialize the countdown timer and log any cached rotation.
        setupCountdownHandler()
        // Optionally retrieve the addedRotation from preferences:
        addedRotation = Helpers.getIntPreferences(CAMERA_ORIENTATION_KEY, 0)
        if (addedRotation != 0) {
            Helpers.printLog("Added Rotation Cached addedRotation: $addedRotation")
        }
    }

    init {
        startProcessingWorker()
    }


    /**
     * Checks if a majority of recent statuses indicate that no face is detected.
     */
    private fun checkForOrientationSuspect() {
        val elapsedTime = System.currentTimeMillis() - lastCheckTime
        if (sessionStatusCount > 0) {
            wrongFaceRatio = wrongFaceRatioCount.toDouble() / sessionStatusCount
            Helpers.printLog("$TAG shouldChangeAngleOrientation: NO_FACE $wrongFaceRatioCount/$sessionStatusCount wrongFaceRatio: $wrongFaceRatio Elapsed: $elapsedTime")
            if (elapsedTime > TIME_PERIOD && wrongFaceRatio > 0.9) {
                faceDetectorListener.onWrongOrientationSuspect()
            }
        }
    }


    private fun startProcessingWorker() {
        processingScope.launch {
            while (isActive) {
                try {
                    val (frame) = processingQueue.poll(100, TimeUnit.MILLISECONDS) ?: continue
                    processFrameInternal(frame)
                } catch (e: Exception) {
                    Log.e("LivenessHandler", "Error processing frame: ${e.message}")
                }
            }
        }
    }

    private fun stop() {
        isStop = true
        processingScope.cancel()
        processingQueue.clear()
        stopTimer()
    }

    fun stopTimer() {
        countDownTimer?.cancel()
    }

    /**
     * Resets orientation check counters.
     */
    private fun refreshFrameCounts() {
        wrongFaceRatioCount = 0
        sessionStatusCount = 0
        checkingForOrientation = false
    }


    fun renewSession() {
        processingQueue.clear()
        livenessSession.renewSession(livenessSessionType)
        refreshFrameCounts()
        sessionStatus = livenessSession.sessionStatus
        startTime = System.currentTimeMillis()
        lastProcessTime = 0L
        isStop = false
        countdown = LIVENESS_MAX_TIME
        setupCountdownHandler()
    }

    /**
     * Sets up and starts the countdown timer.
     */
    private fun setupCountdownHandler() {
        Helpers.printLog("setupCountdownHandler onInit")
        countDownTimer?.cancel()
        if (KalapaSDK.isConfigInitialized() && LIVENESS_MAX_TIME == -1L) return
        countDownTimer = object : CountDownTimer(countdown, 1000L) {
            override fun onTick(millisUntilFinished: Long) {
                faceDetectorListener.onCountdown(millisUntilFinished)
            }

            override fun onFinish() {
                Helpers.printLog("setupCountdownHandler onFinish")
                faceDetectorListener.onExpired()
            }
        }.also { it.start() }
    }

    fun processSession(frame: Bitmap) {
        val currentTime = System.currentTimeMillis()
        if (currentTime - lastProcessTime < PROCESS_INTERVAL) {
            return
        }
        if (processingQueue.size < 2) {
            processingQueue.offer(frame to frame)
        }
    }

    private suspend fun processFrameInternal(frame: Bitmap) {
        val currentTime = System.currentTimeMillis()
        lastProcessTime = currentTime

        val isExpired = currentTime - startTime > LIVENESS_MAX_TIME || countdown == 0L

        sessionStatus = livenessSession.sessionStatus
        sessionAction = livenessSession.getCurrentAction()

        if (!isStop && !livenessSession.isFinished() && !isExpired) {
            if (!checkingForOrientation) {
                checkForOrientationSuspect()
                if (sessionStatus != LivenessSessionStatus.TOO_SMALL &&
                    sessionStatus != LivenessSessionStatus.TOO_LARGE &&
                    sessionStatus != LivenessSessionStatus.OFF_CENTER
                ) {
                    sessionStatusCount++
                }
                if (sessionStatus == LivenessSessionStatus.NO_FACE ||
                    sessionStatus == LivenessSessionStatus.ANGLE_NOT_CORRECT
                ) {
                    wrongFaceRatioCount++
                }
            } else {
                when (sessionStatus) {
                    LivenessSessionStatus.NO_FACE, LivenessSessionStatus.ANGLE_NOT_CORRECT -> {
                        addedRotation = (addedRotation + 90) % 360
                        continuousFaceCounter = 0
                    }

                    LivenessSessionStatus.PROCESSING,
                    LivenessSessionStatus.OFF_CENTER,
                    LivenessSessionStatus.TOO_LARGE,
                    LivenessSessionStatus.TOO_SMALL -> {
                        continuousFaceCounter++
                        if (continuousFaceCounter >= 3) {
                            Helpers.savePrefs(CAMERA_ORIENTATION_KEY, addedRotation)
                            Helpers.printLog("Orientation updated and accepted: $addedRotation")
                            checkingForOrientation = false
                            continuousFaceCounter = 0
                        }
                    }

                    else -> Helpers.printLog("Ignored sessionStatus $sessionStatus")
                }
            }

            actualRotationDegree = (rotationAngle + addedRotation) % 360
            Helpers.printLog("Actual Rotation Degree updated to $actualRotationDegree from $rotationAngle addedRotation $addedRotation")

            withContext(Dispatchers.Default) {
                livenessSession.process(frame, actualRotationDegree, faceDetectorListener)
            }
        } else {
            handleSessionEnd(frame, isExpired)
        }
    }

    private fun handleSessionEnd(frame: Bitmap, isExpired: Boolean) {
        sessionStatus = livenessSession.sessionStatus
        sessionAction = livenessSession.getCurrentAction()

        if (livenessSession.isFinished()) {
            Helpers.printLog("LivenessSessionStatus $sessionStatus addedRotation $addedRotation")
            if (addedRotation != Helpers.getIntPreferences(CAMERA_ORIENTATION_KEY, 0)) {
                Helpers.savePrefs(CAMERA_ORIENTATION_KEY, addedRotation)
                Helpers.printLog("Added Rotation Cached addedRotation: $addedRotation into $CAMERA_ORIENTATION_KEY")
            }

            if (sessionStatus == LivenessSessionStatus.VERIFIED) {
                if (livenessSession.gotTypicalFace) {
                    faceDetectorListener.onFaceDetected(
                        frame,
                        livenessSession.typicalFace,
                        actualRotationDegree
                    )
                }
            } else {
                faceDetectorListener.onMessage(sessionStatus, sessionAction)
            }
        } else if (sessionStatus == LivenessSessionStatus.EXPIRED || isExpired) {
            faceDetectorListener.onExpired()
        }
    }


    fun release() {
        livenessSession.cleanup()
        stop()
        processingQueue.clear()
    }
}
