package vn.kalapa.ekyc.views

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.BlurMaskFilter
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.RectF
import android.util.AttributeSet
import android.view.View
import androidx.interpolator.view.animation.FastOutSlowInInterpolator
import vn.kalapa.ekyc.utils.Common

class OvalLivenessOverlayView @JvmOverloads constructor(
    context: Context,
    attributeSet: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : View(context, attributeSet, defStyleAttr) {

    private val ovalPaint = Paint().apply {
        style = Paint.Style.STROKE
        strokeWidth = dpToPx(2f)
        isAntiAlias = true
    }

    private val overlayPaint = Paint().apply {
        color = Color.parseColor("#BF000000")
        isAntiAlias = true
    }

    private val clearPaint = Paint().apply {
        style = Paint.Style.FILL
        xfermode = PorterDuffXfermode(PorterDuff.Mode.CLEAR)
        isAntiAlias = true
    }

    private val ovalRect = RectF()
    private val blurMaskFilter = BlurMaskFilter(20f, BlurMaskFilter.Blur.NORMAL)
    private val ovalSize = dpToPx(250f)

    private var progressAnimator: ValueAnimator? = null
    private var currentProgress = 0f
    private var currentPart = 1
    private var totalParts = 2
    private var completedProgress = mutableMapOf<Int, Float>()

    private var progressColor = Color.GREEN
    private val backgroundColor = Color.LTGRAY
    private var isLastStepInProgress = false
    private var isAnimating = false

    init {
        overlayPaint.maskFilter = blurMaskFilter
        setLayerType(LAYER_TYPE_HARDWARE, null)
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        canvas.drawRect(0f, 0f, width.toFloat(), height.toFloat(), overlayPaint)

        val centerX = width / 2f
        val centerY = height / 2f
        val ovalWidth = ovalSize
        val ovalHeight = ovalSize * 1.3f

        ovalRect.set(
            centerX - ovalWidth / 2,
            centerY - ovalHeight / 2,
            centerX + ovalWidth / 2,
            centerY + ovalHeight / 2
        )

        // Clear the oval area
        canvas.drawOval(ovalRect, clearPaint)

        // Draw background
        ovalPaint.color = backgroundColor
        ovalPaint.style = Paint.Style.STROKE
        ovalPaint.strokeWidth = dpToPx(2f)
        canvas.drawOval(ovalRect, ovalPaint)

        // Draw progress
        ovalPaint.color = progressColor
        ovalPaint.style = Paint.Style.STROKE
        ovalPaint.strokeWidth = dpToPx(2f)

        val degreesPerPart = 360f / totalParts

        // Draw completed parts
        for (part in 1 until currentPart) {
            val startAngle = -90f + (part - 1) * degreesPerPart
            canvas.drawArc(ovalRect, startAngle, degreesPerPart, false, ovalPaint)
        }

        // Draw current progress
        val startAngle = -90f + (currentPart - 1) * degreesPerPart
        val currentSweepAngle = currentProgress * degreesPerPart
        canvas.drawArc(ovalRect, startAngle, currentSweepAngle, false, ovalPaint)
    }

    private fun startAnim(duration: Long = 2000, onAnimationEnd: (() -> Unit)? = null) {
        progressAnimator?.cancel()

        progressAnimator = ValueAnimator.ofFloat(0f, 1f).apply {
            this.duration = duration
            interpolator = FastOutSlowInInterpolator()
            addUpdateListener { animator ->
                currentProgress = animator.animatedValue as Float
                invalidate()
            }
            addListener(object : AnimatorListenerAdapter() {
                override fun onAnimationEnd(animation: Animator) {
                    completedProgress[currentPart] = 1f
                    isAnimating = false
                    onAnimationEnd?.invoke()
                }
            })
            start()
            isAnimating = true
        }
    }

    fun nextPart(duration: Long = 600) {
        if (currentPart <= totalParts) {
            if (currentPart == totalParts && isLastStepInProgress) {
                return
            }

            progressColor = Color.GREEN

            if (currentPart == totalParts) {
                startSuccessAnimation()
                isLastStepInProgress = true
            }

            startAnim(duration) {
                if (currentPart < totalParts) {
                    currentPart++
                    currentProgress = 0f
                }
                if (currentPart == totalParts) {
                    isLastStepInProgress = false
                }
            }
        }
    }

    fun setLivenessVersion(version: Common.LIVENESS_VERSION) {
        totalParts = when (version) {
            Common.LIVENESS_VERSION.PASSIVE -> 1
            Common.LIVENESS_VERSION.SEMI_ACTIVE -> 2
            Common.LIVENESS_VERSION.ACTIVE -> 3
        }
        currentPart = 1
        completedProgress.clear()
        stopProgress()
    }

    fun progressFail(duration: Long = 2000) {
        progressColor = Color.RED
        currentPart = 0
        totalParts = 1
        completedProgress.clear()
        isLastStepInProgress = false
        startAnim(duration)
    }

    fun stopProgress() {
        progressAnimator?.cancel()
        currentProgress = 0f
        isAnimating = false
        invalidate()
    }

    fun resetProgress() {
        progressAnimator?.cancel()
        currentProgress = 0f
        currentPart = 1
        progressColor = Color.GREEN
        isLastStepInProgress = false
        completedProgress.clear()
        isAnimating = false
        invalidate()
    }

    fun startSuccessAnimation() {
        if (!isAnimating) {
            val successAnimator = ValueAnimator.ofFloat(0f, 1f).apply {
                duration = 500
                addUpdateListener { animator ->
                    val fraction = animator.animatedValue as Float
                    progressColor = interpolateColor(Color.YELLOW, Color.GREEN, fraction)
                    invalidate()
                }
            }
            successAnimator.start()
        }
    }

    private fun interpolateColor(startColor: Int, endColor: Int, fraction: Float): Int {
        val startA = Color.alpha(startColor)
        val startR = Color.red(startColor)
        val startG = Color.green(startColor)
        val startB = Color.blue(startColor)

        val endA = Color.alpha(endColor)
        val endR = Color.red(endColor)
        val endG = Color.green(endColor)
        val endB = Color.blue(endColor)

        return Color.argb(
            (startA + (endA - startA) * fraction).toInt(),
            (startR + (endR - startR) * fraction).toInt(),
            (startG + (endG - startG) * fraction).toInt(),
            (startB + (endB - startB) * fraction).toInt()
        )
    }

    private fun dpToPx(dp: Float): Float {
        return dp * context.resources.displayMetrics.density
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        progressAnimator?.cancel()
    }
}