package wang.report.starter;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

import wang.report.querier.brige.DataBrige;
import wang.report.querier.brige.SecurityBrige;
import wang.report.querier.domain.Result;
import wang.report.querier.domain.ResultCodes;
import wang.report.querier.domain.SimpleColumn;
import wang.report.querier.domain.SimpleReport;
import wang.report.querier.domain.SimpleReportRuntime;
import wang.report.querier.format.FormaterProvider;
import wang.report.querier.querier.ReportLoader;
import wang.report.querier.querier.ReportQuerier;
import wang.report.querier.util.Common;
/**
 * 一个报表Ctroller的基础实现，默认从文件系统中读取报表定义，文件系统的位置可通过report.dir配置，如果不希望从文件系统读取，但仍想使用其余功能，可以重写readReportDefine方法
 * @author wangjiejin
 *
 */
@RestController
public abstract class DefaultReportController {
	
	private static final Logger log = LoggerFactory.getLogger(DefaultReportController.class);

	@Autowired
	private SecurityBrige securityBrige;
	@Autowired
	private DataBrige dataBrige;
	@Autowired
	private FormaterProvider formaterProvider;
	@Autowired
	private DataSource dataSource;
	@Value("${report.dir:reports}")
	private String saveDir="";
	
	/**
	 * 读取报表的定义和运行时数据
	 * @param reportId
	 * @return
	 */
	@GetMapping("{reportId}")
	public Result<SimpleReportRuntime, ?> load(@PathVariable("reportId") String reportId) {
		Result<SimpleReportRuntime, ?> result = new Result<>();
		SimpleReport report = readReportDefine(reportId);
		if (report == null) {
			return result.setCode(ResultCodes.RESOURCE_NOT_FOUND).setReason("报表"+reportId+"不存在");
		}
		try (ReportLoader loader = new ReportLoader()) {
			loader.setSecurityBrige(securityBrige);
			loader.setDataBrige(dataBrige);
			loader.setDataSource(dataSource);
			return loader.loadReport(report);
		} catch (Exception e) {
			return result.setCode(ResultCodes.INTENAL_ERROR).setReason("加载报表时发生内部错误");
		}
		
	}
	
	/**
	 * 执行查询
	 * @param map 数据参数。key:如果过滤字段定义了alias，则使用alias，没有则使用columnName；value:单值直接传值，in可以使用数组，或者逗号分割，范围可以使用数组
	 * @param reportId 报表ID
	 * @return 报表数据
	 */
	@RequestMapping(path="/query/{reportId}", method=RequestMethod.POST)
	public Result<List<Map<String, Object>>, Object> query(@RequestBody Map<String, Object> map, @PathVariable("reportId") String reportId) {
		SimpleReport report = readReportDefine(reportId);
		if (report == null) {
			return new Result<List<Map<String, Object>>, Object>().setCode(ResultCodes.RESOURCE_NOT_FOUND).setReason("报表"+reportId+"不存在");
		}
		return query(map, report, false);
	}
	
	/**
	 * 执行查询和导出EXCEL
	 * @param map map 数据参数。key:如果过滤字段定义了alias，则使用alias，没有则使用columnName；value:单值直接传值，in可以使用数组，或者逗号分割，范围可以使用数组
	 * @param reportId reportId 报表ID
	 * @param response
	 * @throws IOException
	 */
	@RequestMapping("/excel/{reportId}")
	public void exportExcel(@RequestBody Map<String, Object> map, @PathVariable("reportId") String reportId, HttpServletResponse response) throws IOException {
		SimpleReport report = readReportDefine(reportId);
		if (report == null) {
			response.sendError(404, "报表"+reportId+"不存在");
			return;
		}
		Result<List<Map<String, Object>>, Object> result = query(map, report, true);
		if (!result.success()) {
			response.sendError(500, result.getReason());
			return;
		}
		report.getColumns().removeIf(c->Common.notEmpty(c.getPermit()) && !securityBrige.isPermitted(c.getPermit()) || c.getHidden());
		response.setContentType("application/vnd.ms-excel");
        response.setHeader("Content-Disposition",
                "attachment;filename=" + reportId+".xlsx");
		try (OutputStream out = response.getOutputStream()) {
			writeListToExcel(out, report.getColumns(), result.getData());
		}
		
	}
		
	private Result<List<Map<String, Object>>, Object> query(Map<String, Object> map, SimpleReport report, boolean exportModel) {		
		if (report == null) {
			return new Result<List<Map<String, Object>>, Object>().setCode(ResultCodes.RESOURCE_NOT_FOUND).setReason("报表不存在");
		}
		if (!Common.isEmpty(report.getReportPermit()) && !securityBrige.isPermitted(report.getReportPermit())) {
			//没有权限
			return new Result<List<Map<String, Object>>, Object>().setCode(ResultCodes.NOT_AUTHORIED).setReason("没有权限打开此报表"+report.getReportId());
		}
		Integer curPage = 1;
		Integer rowsPerPage = Integer.MAX_VALUE;	
		if (!exportModel) {
			curPage = saftyGet(map.get("page"), 1);
			rowsPerPage = saftyGet(map.get("rows"), 20);
		}		
		try (ReportQuerier querier = new ReportQuerier()) {			
			querier.setDataSource(dataSource);
			querier.setDataBrige(dataBrige);
			querier.setFormaterProvider(formaterProvider);
			querier.setSecurityBrige(securityBrige);
			return querier.set(report, map, rowsPerPage, curPage).query();
		} catch (SQLException e) {
			log.error("查询发生异常", e);
		}
		return new Result<List<Map<String, Object>>, Object>().setCode(ResultCodes.INTENAL_ERROR).setReason("报表查询出错了");
	}
	
	/**
	 * 读取报表的设计定义
	 * @param reportId 报表ID
	 * @return
	 */
	protected SimpleReport readReportDefine(String reportId) {
		ObjectMapper mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES, false);
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		try {
			return mapper.readValue(new File(saveDir, reportId), SimpleReport.class);
		} catch (IOException e) {			
			log.error("没有找到指定的报表：" + reportId+"位于"+saveDir);
		}
		return null;
//		return reportRepo.readContent(reportId);
	}
	
	/**
	 * 数据导出EXCEL的实现，默认未提供实现，如要导出，则需要重写
	 * @param out 输出流
	 * @param columns 列定义
	 * @param data 数据
	 */
	protected void writeListToExcel(OutputStream out, List<SimpleColumn> columns, List<Map<String, Object>> data) {
		throw new RuntimeException("默认未提供导出EXCEL的实现");
	}
	
	private Integer saftyGet(Object input, Integer min) {
		if (input == null || input instanceof String && Common.isEmpty((String)input)) {
			return min;
		}
		Integer i = null;
		if (input instanceof Integer) {
			i = (Integer)input;
		} else {			
			try {
				i = Integer.valueOf(input.toString());
			} catch (Exception e) {
				i = 0;
			}
		}
		return Math.max(i, min);
	}


	public String getSaveDir() {
		return saveDir;
	}


	public void setSaveDir(String saveDir) {
		this.saveDir = saveDir;
	}

}
