/*
 * $Id: SortableListViewHeaders.java 461192 2006-06-28 06:37:16Z ehillenius $
 * $Revision: 461192 $ $Date: 2006-06-28 08:37:16 +0200 (Wed, 28 Jun 2006) $
 * 
 * ==================================================================== Licensed
 * under the Apache License, Version 2.0 (the "License"); you may not use this
 * file except in compliance with the License. You may obtain a copy of the
 * License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package wicket.examples.displaytag.list;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import wicket.MarkupContainer;
import wicket.markup.ComponentTag;
import wicket.markup.MarkupStream;
import wicket.markup.html.WebMarkupContainer;
import wicket.markup.html.list.ListView;
import wicket.markup.resolver.IComponentResolver;

/**
 * This is a convenient component to create sorted list view headers very
 * easily. It first scans the markup for &lt;th wicket:id=".*" ..&gt> tags and
 * automatically creates a SortableListViewHeader for each.
 * <p>
 * The component can only be used with &lt;thead&gt; tags.
 * 
 * @see SortableListViewHeaderGroup
 * @see SortableListViewHeader
 * @author Juergen Donnerstag
 */
public class SortableListViewHeaders extends WebMarkupContainer implements IComponentResolver
{
	/** Logging. */
	private static final Log log = LogFactory.getLog(SortableListViewHeaders.class);

	/** Each SortableTableHeader (without 's) must be attached to a group. */
	final private SortableListViewHeaderGroup group;

	/**
	 * Construct.
	 * 
	 * @param id
	 *            The component's id; must not be null
	 * @param listView
	 *            the underlying ListView
	 */
	public SortableListViewHeaders(final String id, final ListView listView)
	{
		super(id);

		this.group = new SortableListViewHeaderGroup(this, listView);
	}

	/**
	 * Compare two object of the column to be sorted, assuming both Objects
	 * support compareTo().
	 * 
	 * @see Comparable#compareTo(java.lang.Object)
	 * @param header
	 * @param o1
	 * @param o2
	 * @return compare result
	 */
	protected int compareTo(final SortableListViewHeader header, final Object o1, final Object o2)
	{
		Comparable obj1 = getObjectToCompare(header, o1);
		Comparable obj2 = getObjectToCompare(header, o2);
		return obj1.compareTo(obj2);
	}

	/**
	 * Get one of the two Object to be compared for sorting a column.
	 * 
	 * @param header
	 * @param object
	 * @return comparable object
	 */
	protected Comparable getObjectToCompare(final SortableListViewHeader header, final Object object)
	{
		return (Comparable)object;
	}

	/**
	 * 
	 * @param container
	 * @param markupStream
	 * @param tag
	 * @return true, if component got resolved
	 */
	public boolean resolve(final MarkupContainer container, final MarkupStream markupStream,
			final ComponentTag tag)
	{
		if (tag.getName().equalsIgnoreCase("th"))
		{
			// Get component name
			final String componentId = tag.getId();
			if ((componentId != null) && (get(componentId) == null))
			{
				autoAdd(new SortableListViewHeader(componentId, group)
				{
					protected int compareTo(final Object o1, final Object o2)
					{
						return SortableListViewHeaders.this.compareTo(this, o1, o2);
					}

					protected Comparable getObjectToCompare(final Object object)
					{
						return SortableListViewHeaders.this.getObjectToCompare(this, object);
					}
				});
				return true;
			}
		}

		return false;
	}

	/**
	 * Scan the related markup and attach a SortableListViewHeader to each
	 * &lt;th&gt; tag found.
	 * 
	 * @see wicket.Component#onRender(MarkupStream)
	 */
	protected void onRender(final MarkupStream markupStream)
	{
		// Must be <thead> tag
		ComponentTag tag = markupStream.getTag();
		checkComponentTag(tag, "thead");

		// Continue with default behavior
		super.onRender(markupStream);
	}
}
