/*
 * $Id$ $Revision$ $Date$
 * 
 * ==============================================================================
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package wicket.extensions.markup.html.tree.table;

import javax.swing.tree.TreeNode;

import wicket.Component;
import wicket.MarkupContainer;
import wicket.Response;
import wicket.util.string.Strings;

/**
 * Convenience class for creating non-interactive lightweight (IRenderable
 * based) columns.
 * 
 * @author Matej Knopp
 */
public abstract class AbstractRenderableColumn extends AbstractColumn
{
	private boolean contentAsTooltip = false;

	private boolean escapeContent = true;

	/**
	 * Creates the column
	 * 
	 * @param location
	 *            Specifies how the column should be aligned and what his size
	 *            should be
	 * 
	 * @param header
	 *            Header caption
	 */
	public AbstractRenderableColumn(ColumnLocation location, String header)
	{
		super(location, header);
	}

	/**
	 * Returns the string value for the provided node.
	 * 
	 * @param node
	 *            Determines the position in tree
	 * @return The
	 */
	public abstract String getNodeValue(TreeNode node);

	/**
	 * Returns whether the content should also be visible as tooltip of the
	 * cell.
	 * 
	 * @return whether the content should also be visible as tooltip
	 */
	public boolean isContentAsTooltip()
	{
		return contentAsTooltip;
	}

	/**
	 * Returns whether the special html characters of content will be escaped.
	 * 
	 * @return Whether html characters should be escaped
	 */
	public boolean isEscapeContent()
	{
		return escapeContent;
	}

	/**
	 * @see IColumn#newCell(MarkupContainer, String, TreeNode, int)
	 */
	public Component newCell(MarkupContainer parent, String id, TreeNode node, int level)
	{
		return null;
	}

	/**
	 * @see IColumn#newCell(TreeNode, int)
	 */
	public IRenderable newCell(TreeNode node, int level)
	{
		return new IRenderable()
		{
			private static final long serialVersionUID = 1L;

			public void render(TreeNode node, Response response)
			{
				String content = getNodeValue(node);

				// escape if necessary
				if (isEscapeContent())
				{
					content = Strings.escapeMarkup(content).toString();
				}

				response.write("<span");
				if (isContentAsTooltip())
				{
					response.write(" title=\"" + content + "\"");
				}
				response.write(">");
				response.write(content);
				response.write("</span>");
			}
		};
	}

	/**
	 * Sets whether the content should also be visible as tooltip (html title
	 * attribute) of the cell.
	 * 
	 * @param contentAsTooltip
	 *            whether the content should also be visible as tooltip
	 */
	public void setContentAsTooltip(boolean contentAsTooltip)
	{
		this.contentAsTooltip = contentAsTooltip;
	}

	/**
	 * Sets whether the special html characters of content should be escaped.
	 * 
	 * @param escapeContent
	 *            Whether to espcape html characters
	 */
	public void setEscapeContent(boolean escapeContent)
	{
		this.escapeContent = escapeContent;
	}
}
