package xin.xihc.utils.common;

import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.concurrent.ConcurrentTaskScheduler;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;
import org.springframework.scheduling.support.CronTrigger;

import java.util.Date;
import java.util.concurrent.*;

/**
 * 任务调度工具类
 * 修正：支持自动扩容核心线程数
 *
 * @author Leo.Xi
 * @date 2019/11/11
 * @since 1.0
 **/
public final class ScheduleUtils {

    /** 核心线程数 */
    private static int CORE_THREAD_COUNT = 0;

    /** 线程池 */
    private static final ScheduledExecutorService EXECUTOR =
            Executors.newScheduledThreadPool(CORE_THREAD_COUNT, new CustomizableThreadFactory("schedule-pool-"));

    /** 任务调用器 */
    private static final ConcurrentTaskScheduler TASK_SCHEDULER = new ConcurrentTaskScheduler(EXECUTOR);

    /** 任务管理器 */
    private static final ConcurrentHashMap<String, ScheduledFuture<?>> TASK_MANAGER = new ConcurrentHashMap<>(256);

    private ScheduleUtils() {
    }

    /**
     * 设置任务调度执行器 - 线程池
     *
     * @param scheduledExecutor 线程池Executor
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void setScheduledExecutor(ScheduledExecutorService scheduledExecutor) {
        TASK_SCHEDULER.setScheduledExecutor(scheduledExecutor);
    }

    /**
     * 设置线程池的核心线程数
     *
     * @param corePoolSize 核心线程数（只能）
     * @return
     * @author Leo Xi
     * @date 2020/5/27
     * @since 0.0.1
     */
    public static void setCorePoolSize(int corePoolSize) {
        if (EXECUTOR instanceof ThreadPoolExecutor) {
            ((ThreadPoolExecutor) EXECUTOR).setCorePoolSize(corePoolSize);
        }
    }

    /**
     * 检测当前任务数量并自动扩容
     *
     * @author Leo Xi
     * @date 2020/5/27
     * @since 0.0.1
     */
    private static void checkCoreTheadCount() {
        if (TASK_MANAGER.size() > CORE_THREAD_COUNT) {
            ++CORE_THREAD_COUNT;
            setCorePoolSize(CORE_THREAD_COUNT);
        }
    }

    /**
     * 新增任务（key重复的则会覆盖）
     *
     * @param key            id标识
     * @param cronExpression cron表达式
     * @param task           任务
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void putTask(String key, String cronExpression, Runnable task) {
        newTask(key, new CronTrigger(cronExpression), task, true);
    }

    /**
     * 新增任务（key重复的则会抛出IllegalArgumentException异常）
     *
     * @param key            id标识
     * @param cronExpression cron表达式
     * @param task           任务
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void addTask(String key, String cronExpression, Runnable task) {
        newTask(key, new CronTrigger(cronExpression), task, false);
    }

    /**
     * 新增任务
     *
     * @param key      id标识
     * @param trigger  cron表达式
     * @param task     任务
     * @param override 是否覆盖
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    private static void newTask(String key, Trigger trigger, Runnable task, boolean override) {
        ScheduledFuture<?> scheduledFuture = TASK_MANAGER.get(key);
        if (scheduledFuture != null) {
            if (override) {
                scheduledFuture.cancel(true);
            } else { // 不覆盖则抛出异常
                throw new IllegalArgumentException("key exists");
            }
        }
        ScheduledFuture<?> schedule = TASK_SCHEDULER.schedule(task, trigger);
        TASK_MANAGER.put(key, schedule);
        checkCoreTheadCount();
    }

    /**
     * 新增一次性任务（只会执行一次）
     *
     * @param key       key-ID
     * @param startTime 开始执行的时间点
     * @param task      任务
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void putTaskOnce(String key, Date startTime, Runnable task) {
        ScheduledFuture<?> scheduledFuture = TASK_MANAGER.get(key);
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
        }
        ScheduledFuture<?> schedule = TASK_SCHEDULER.schedule(task, startTime);
        TASK_MANAGER.put(key, schedule);
        checkCoreTheadCount();
    }

    /**
     * 新增定时周期性任务
     * 这个是以period为固定周期时间，按照一定频率来重复执行任务。
     * 举个栗子：高铁定时发车，过时不候；period为一天，以天为周期，优先保证任务执行的频率。
     *
     * @param key       key-ID
     * @param startTime 开始执行的时间点
     * @param period    以后执行的间隔时间（周期-毫秒值）
     * @param task      任务
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void putTaskAtFixedRate(String key, Date startTime, long period, Runnable task) {
        ScheduledFuture<?> scheduledFuture = TASK_MANAGER.get(key);
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
        }
        ScheduledFuture<?> schedule = TASK_SCHEDULER.scheduleAtFixedRate(task, startTime, period);
        TASK_MANAGER.put(key, schedule);
        checkCoreTheadCount();
    }

    /**
     * 新增定时周期性任务
     * 这个是优先保证任务执行的间隔。
     * 这个是以delay为固定延迟时间，按照一定的等待时间来执行任务。
     *
     * @param key       key-ID
     * @param startTime 开始执行的时间点
     * @param period    以后执行的间隔时间（周期-毫秒值）
     * @param task      任务
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void putTaskWithFixedDelay(String key, Date startTime, long period, Runnable task) {
        ScheduledFuture<?> scheduledFuture = TASK_MANAGER.get(key);
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
        }
        ScheduledFuture<?> schedule = TASK_SCHEDULER.scheduleWithFixedDelay(task, startTime, period);
        TASK_MANAGER.put(key, schedule);
        checkCoreTheadCount();
    }

    /**
     * 移除任务
     *
     * @param key key-ID
     * @author Leo.Xi
     * @date 2019/11/12
     * @since 0.0.1
     */
    public static void removeTask(String key) {
        ScheduledFuture<?> scheduledFuture = TASK_MANAGER.get(key);
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
            TASK_MANAGER.remove(key);
        }
    }

    /**
     * 是否存在任务
     *
     * @param key 任务KEY
     * @return true-存在，false-不存在
     * @author Leo Xi
     * @date 2020/9/22
     * @since 0.0.1
     */
    public static boolean existsTask(String key) {
        return TASK_MANAGER.containsKey(key);
    }

}