package xin.xihc.utils.log;

import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.PatternLayout;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.ConsoleAppender;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.rolling.SizeAndTimeBasedRollingPolicy;
import ch.qos.logback.core.util.FileSize;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xin.xihc.utils.common.ScheduleUtils;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedList;

/**
 * 保存日志记录到文件,支持多线程操作,支持多个文件保存,限制单个文件大小
 *
 * @author Leo Xi
 * @date 2020/5/25
 * @since 1.20.2
 **/
public class LogUtil {

    public static final String SCHEDULE_NAME = "LOG_0_0_INIT_FILE_APPENDER";
    private static final String CONSOLE_APPENDER_NAME = "logToConsole";

    /***********************************************************************************
     *
     *              如需要修改的则需要在程序启动时设置好
     *      （后面设置几乎无效 - 因为logger都是启动时都已经加载好的）
     *
     **********************************************************************************/
    /** 默认保存 60 天的日志 */
    public static int MAX_HISTORY = 60;
    /** 单个日志文件袋小 - 默认10MB（GB、MB、KB） */
    public static String MAX_FILE_SIZE = "10MB";
    /** 日志总大小 - 默认20GB（GB、MB、KB） */
    public static String TOTAL_SIZE = "20GB";
    /** 日志文件夹 */
    public static String LOG_DIR = "./logs/%d{yyyyMM}/%d{dd}";
    /** 日志内容的格式 */
    public static String LOG_MSG_PATTERN = "%d{yyyy-MM-dd HH:mm:ss.SSS} [%thread] %-5level %class::[%method:%L] %n%msg%n%n";

    /** 是否同时输出到控制台 - 默认 true */
    private static boolean TO_CONSOLE = true;
    private static LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
    private static LinkedList<ch.qos.logback.classic.Logger> LOGS = new LinkedList<>();
    /** 文件名一致时APPENDER保持一样 */
    private static LinkedHashMap<String, RollingFileAppender<ILoggingEvent>> APPENDER = new LinkedHashMap<>();

    static {
        resetRoot();
    }

    /**
     * 每天0点0分1秒的时候重新设置文件路径&文件名
     *
     * @author Leo Xi
     * @date 2020/6/3
     * @since 0.0.1
     */
    public static synchronized void initFileAppender() {
        System.err.println("开始重新设置文件路径" + new Date().toString());
        // 更新新的文件Appender
        String[] names = APPENDER.keySet().toArray(new String[0]);
        APPENDER.clear();
        for (String k : names) {
            RollingFileAppender<ILoggingEvent> appender = getFileAppender(k);
            APPENDER.put(k, appender);
            LOGS.forEach(l -> {
                // 移除对应的appender
                Appender<ILoggingEvent> old = l.getAppender(k);
                if (old != null) {
                    l.detachAndStopAllAppenders();
                    l.addAppender(appender);
                    appender.start();
                }
            });
        }
        setToConsole(TO_CONSOLE);
    }

    /**
     * 重新设置Root
     *
     * @author Leo Xi
     * @date 2020/6/3
     * @since 0.0.1
     */
    public static synchronized void resetRoot() {
        ch.qos.logback.classic.Logger root = loggerContext.getLogger(Logger.ROOT_LOGGER_NAME);
        root.detachAndStopAllAppenders();
        root.addAppender(getConsoleAppender());
    }

    /**
     * 获取logger对象
     *
     * @param clazz 类型
     * @return org.slf4j.Logger
     * @author Leo Xi
     * @date 2020/5/25
     * @since 1.20.2
     */
    public static Logger getLogger(Class<?> clazz) {
        return getLogger(clazz.getName(), null);
    }

    /**
     * 获取logger对象
     *
     * @param clazz       类型
     * @param logFileName 日志对应的文件名称
     * @return org.slf4j.Logger
     * @author Leo Xi
     * @date 2020/5/25
     * @since 1.20.2
     */
    public static Logger getLogger(Class<?> clazz, String logFileName) {
        return getLogger(clazz.getName(), logFileName);
    }

    /**
     * 获取logger对象
     *
     * @param logFileName 日志文件名称 对应的log name是 log.{logFileName}
     * @return org.slf4j.Logger
     * @author Leo Xi
     * @date 2020/5/25
     * @since 1.20.2
     */
    public static Logger getLogger(String logFileName) {
        if (null == logFileName || "".equals(logFileName.trim())) {
            logFileName = "info";
        }
        return getLogger("log." + logFileName, logFileName);
    }

    /**
     * 获取logger对象
     *
     * @param logName     日志名称
     * @param logFileName 日志对应的文件名称
     * @return org.slf4j.Logger
     * @author Leo Xi
     * @date 2020/5/25
     * @since 1.20.2
     */
    public static synchronized Logger getLogger(String logName, String logFileName) {
        if (null == logFileName || "".equals(logFileName.trim())) {
            logFileName = "info";
        }
        ch.qos.logback.classic.Logger newLog = loggerContext.getLogger(logName);
        logFileName = "log-" + logFileName;
        LOGS.add(newLog);
        newLog.detachAndStopAllAppenders();

        newLog.addAppender(getFileAppender(logFileName));
        newLog.setAdditive(false); // 不继承父类的输出

        // 输出到console
        if (TO_CONSOLE) {
            newLog.setAdditive(true); // 继承父类的输出
        }
        if (!ScheduleUtils.existsTask(SCHEDULE_NAME)) {
            // 每天0点0分0秒的时候重新设置文件路径&文件名 0 0 0 * * *
            ScheduleUtils.putTask(SCHEDULE_NAME, "1 0 0 * * *", LogUtil::initFileAppender);
        }
        return newLog;
    }

    /**
     * 获取输出到文件的Appender
     *
     * @return RollingFileAppender<ILoggingEvent>
     * @author Leo.Xi
     * @date 2020/5/31
     * @since 0.0.1
     */
    private static RollingFileAppender<ILoggingEvent> getFileAppender(String logFileName) {
        return APPENDER.computeIfAbsent(logFileName, k -> {
            RollingFileAppender<ILoggingEvent> fileAppender = new RollingFileAppender<>();

            SizeAndTimeBasedRollingPolicy<ILoggingEvent> policy = new SizeAndTimeBasedRollingPolicy<>();
            policy.setContext(loggerContext);
            policy.setMaxHistory(MAX_HISTORY);
            policy.setTotalSizeCap(FileSize.valueOf(TOTAL_SIZE));
            policy.setFileNamePattern(LOG_DIR + "/" + logFileName + ".%i.log");
            policy.setParent(fileAppender);
            policy.setMaxFileSize(FileSize.valueOf(MAX_FILE_SIZE));
            policy.start();

            //encoder
            PatternLayout layout = new PatternLayout();
            layout.setContext(loggerContext);
            layout.setPattern(LOG_MSG_PATTERN);
            layout.start();

            //start appender
            fileAppender.setRollingPolicy(policy);
            fileAppender.setName(logFileName); // 设置appender的name为文件名
            fileAppender.setContext(loggerContext);
            fileAppender.setLayout(layout);
            fileAppender.setPrudent(true); //support that multiple JVMs can safely write to the same file.
            fileAppender.start();

            return fileAppender;
        });
    }

    /**
     * 获取输出到控制台的Appender
     *
     * @return ConsoleAppender<ILoggingEvent>
     * @author Leo.Xi
     * @date 2020/5/31
     * @since 0.0.1
     */
    private static ConsoleAppender<ILoggingEvent> getConsoleAppender() {
        ConsoleAppender<ILoggingEvent> consoleAppender = new ConsoleAppender<>();

        PatternLayout layout = new PatternLayout();
        layout.setContext(loggerContext);
        layout.setPattern(LOG_MSG_PATTERN);
        layout.start();

        consoleAppender.setContext(loggerContext);
        consoleAppender.setLayout(layout);
        consoleAppender.setName(CONSOLE_APPENDER_NAME);
        consoleAppender.start();

        return consoleAppender;
    }

    /**
     * 动态设置是否输出到控制台
     *
     * @param toConsole true-输出到控制台；false-不输出到控制台
     * @author Leo.Xi
     * @date 2020/5/31
     * @since 0.0.1
     */
    public static synchronized void setToConsole(boolean toConsole) {
        TO_CONSOLE = toConsole;

        for (ch.qos.logback.classic.Logger log : LOGS) {
            if (toConsole) {
                log.setAdditive(true);
            } else {
                log.setAdditive(false);
            }
        }
    }
}