/* 
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.autoclose;


import java.io.Closeable;
import java.lang.ref.Reference;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Автоматическое закрытие объектов. <br>
 * Создает поток (имя AutoCloser) для закрытия обеъктов к конце работы jre. <br>
 * Поддерживаются объекты следующих типов:
 * <ul>
 * <li>java.lang.ref.Reference и производные, для получения целевой ссылки</li>
 * <li>java.io.Closeable и производные</li>
 * <li>java.io.Runnable и производные</li>
 * <li>java.sql.Connection и производные</li>
 * <li>java.sql.ResultSet и производные</li>
 * <li>java.sql.Statement и производные</li>
 * </ul>
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class AutoCloser {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(AutoCloser.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    private final Lock lock = new ReentrantLock();
    
    protected AutoCloser(){
        initThread();
    }
    
    protected static AutoCloser instance;
    
    public synchronized static AutoCloser get(){
        if( instance!=null )return instance;
        if( isLogFine )logFine("create AutoCloser instance");
        instance = new AutoCloser();
        return instance;
    }
    
    private Collection objects;
    
    protected Collection objects(){
        try {
            lock.lock();
            if( objects!=null )return objects;
            objects = Collections.synchronizedSet( new LinkedHashSet() );
            if( isLogFine ){
                logFine("created collection for release objects");
            }
            return objects;
        }
        finally {
            lock.unlock();
        }
    }
    
    public void add( Object obj ){
        if( obj==null )return;
        if( isLogFine )logFine("add( {0} )", obj);
        try {
            lock.lock();
            Collection c = objects();
            c.add(obj);
        }
        finally {
            lock.unlock();
        }
    }
    
    public void remove( Object obj ){
        if( obj==null )return;
        if( isLogFine )logFine("remove( {0} )", obj);
        try {
            lock.lock();
            Collection c = objects();
            c.remove(obj);
        }
        finally {
            lock.unlock();
        }
    }

    public void closeAll(){
        try {
            if( isLogFine )logFine("closeAll( {0} )");
            
            lock.lock();
            Collection cobjs = objects();
            
            if( isLogFiner )logFiner("size of queue = {0}", cobjs.size());
            
            int idx = 0;
            int cnt = cobjs.size();
            
            for( Object o : cobjs ){
                idx++;
                
                if( o==null )continue;
                
                if( isLogFiner )logFiner("{0}/{1} close", idx, cnt);
                
                Object co = o;
                if( co instanceof Reference ){
                    co = ((Reference)co).get();
                }
                
                try{
                    if( co instanceof Closeable ){
                        ((Closeable)co).close();
                    }else if( co instanceof Runnable ){
                        ((Runnable)co).run();
                    }else if( co instanceof java.sql.Connection ){
                        ((java.sql.Connection)co).close();
                    }else if( co instanceof java.sql.ResultSet ){
                        ((java.sql.ResultSet)co).close();
                    }else if( co instanceof java.sql.Statement ){
                        ((java.sql.Statement)co).close();
                    }
                }
                catch( Throwable err ){
                    if( isLogSevere )logger.log(Level.SEVERE, "fail on close "+co, err);
                }
            }
            
            if( isLogFiner )logFiner("clear collection of release objects");
            cobjs.clear();
        }
        finally {
            lock.unlock();
        }
    }
    
    private static Thread thread;
    
    protected void initThread(){
        if( thread!=null )return;
        if( isLogFine )logFine("initThread");

        thread = new Thread(){
            @Override
            public void run() {
                AutoCloser.get().closeAll();
            }
        };
        thread.setName("AutoCloser");
        
        Runtime.getRuntime().addShutdownHook(thread);
    }
}
