/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.cli;


import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.tree.AbstractTreeNode;
import xyz.cofe.text.parser.lex.FieldAccess;
import xyz.cofe.typeconv.ExtendedCastGraph;
import xyz.cofe.typeconv.TypeCastGraph;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class AssignVar
extends AbstractTreeNode<Value>
implements Value, DefineVar
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(AssignVar.class.getName());
    private static final Level logLevel = logger.getLevel();

    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock = new ReentrantLock();

    public AssignVar(){
    }

    //<editor-fold defaultstate="collapsed" desc="children">
    @Override
    public Value[] getChildren() {
        if( data==null )return new Value[]{};
        return new Value[]{ data };
//        return new Value[]{};
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="memory">
    private Memory memory;

    public Memory getMemory() {
        return memory;
    }

    public void setMemory(Memory memory) {
        this.memory = memory;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="data">
    private Value  data;
    public Value getData() {
        return data;
    }

    public void setData(Value data) {
        this.data = data;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="field">
    private FieldAccess field;
    public FieldAccess getField() {
        return field;
    }

    @Override
    public String getVarName() {
        if( field==null )return null;
        return field.toString();
    }

    public void setField(FieldAccess field) {
        this.field = field;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="type">
    @Override
    public Class getType() {
        if( data==null )return Void.class;
        return data.getType();
    }

    @Override
    public Class getVarType() {
        if( data==null )return Void.class;
        return data.getType();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="assignedValue">
    protected Object assignedValue;

    public Object getAssignedValue() {
        return assignedValue;
    }

    public void setAssignedValue(Object assignedValue) {
        this.assignedValue = assignedValue;
    }

    @Override
    public Object getVarValue() {
        return assignedValue;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="typeCastGraph">
    protected TypeCastGraph typeCastGraph;

    public TypeCastGraph getTypeCastGraph() {
        try{
            lock.lock();
            if( typeCastGraph!=null ){
                return typeCastGraph;
            }
            typeCastGraph = new ExtendedCastGraph();
            return typeCastGraph;
        }finally{
            lock.unlock();
        }
    }

    public void setTypeCastGraph(TypeCastGraph typeCastGraph) {
        try{
            lock.lock();
            this.typeCastGraph = typeCastGraph;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    protected String getCodePath(){
        String codepath = "";
        Value v = this;
        while( v!=null ){
            int idx = v.getIndex();
            String cname = v.getClass().getName();
            String nodeText = Integer.toString(idx)+":"+cname;
            if( codepath.length()>0 ){
                codepath = nodeText+" / "+codepath;
            }else{
                codepath = nodeText;
            }
            v = v.getParent();
        }
        return codepath;
    }

    @Override
    public Object eval() {
        if( data==null ){
            return null;
        }
        if( memory==null ){
            return null;
        }

        Object v = data.eval();

        if( field==null ){
//            return null;
            throw new Error( "can't assign var value, field not set, code="+getCodePath() );
        }else{
            String[] path = field.getAccessPath();
            if( path==null )return null;
            if( path.length==0 )return null;
            if( path.length==1 ){
                if( memory.getConsts().contains(path[0]) ){
                    throw new Error("can't assign var "+path[0]+" - variable is readonly");
                }else{
                    memory.put(path[0], v);
                    assignedValue = v;
                    return v;
                }
            }
            String err = assignStart(memory, path, v, new Func1<Object, Object>() {
                @Override
                public Object apply(Object arg) {
                    assignedValue = arg;
                    return null;
                }
            });
            if( err==null ){
//                assignedValue = v;
                return assignedValue;
            }else{
                assignedValue = null;
            }
            throw new Error( "can't assign var value, field not set, code="+getCodePath()+", "+err );
        }
//        return null;
    }

    protected String assignStart( Memory mem, String[] path, Object value, Func1<Object,Object> assigned ){
        if( mem==null )return "memory not set";
        if( path==null )return "var name (path) is null";
        if( path.length==0 )return "var name (path) is empty";
        if( path.length==1 ){
            if( mem.getConsts().contains(path[0]) ){
                throw new Error("can't assign var "+path[0]+" - variable is readonly");
            }
            mem.put(path[0], value);
            if( assigned!=null ){
                assigned.apply(value);
            }
            return null;
        }

        Object root = mem.get(path[0]);
        if( root==null )return "var "+path[0]+" not found";

        String[] newpath = new String[path.length-1];
        for( int i=1; i<path.length; i++ ){
            newpath[i-1] = path[i];
        }

        return assign(root, newpath, value, assigned);
    }

    protected String assign( Object root, String[] path, Object value, Func1<Object,Object> assigned ){
        if( root==null )return "null reference";
        if( path==null )return "var name (path) is null";
        if( path.length==0 )return "var name (path) is empty";

        BeanInfo bi = null;
        try {
            bi = Introspector.getBeanInfo(root.getClass());
        } catch (IntrospectionException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        }

        if( path.length==1 ){
            return assign(root, path[0], value, assigned);
        }

        Object newroot = null;

        try {
            for( PropertyDescriptor pd : bi.getPropertyDescriptors() ){
                if( !pd.getName().equals(path[0]) )continue;

                if( pd.getPropertyType()==null ){
                    return "may be index property";
                }

                Method m = pd.getReadMethod();
                if( m==null )return "write only property";

                newroot = m.invoke(root);
                break;
            }

            if( newroot==null ){
                for( Field f : root.getClass().getFields() ){
                    if( !f.getName().equals(path[0]) )continue;
                    newroot = f.get(root);
                    break;
                }
            }

            if( root instanceof Map ){
                newroot = ((Map)root).get(path[0]);
            }
        } catch (IllegalAccessException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        } catch (InvocationTargetException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        }
        if( newroot==null ){
            return "property "+path[0]+" not found";
        }

        String[] newpath = new String[path.length-1];
        for( int i=1; i<path.length; i++ ){
            newpath[i-1] = path[i];
        }

        return assign(newroot, newpath, value, assigned);
    }

    protected String assign( Object root, String property, Object value, Func1<Object,Object> assigned ){
        BeanInfo bi = null;
        try {
            bi = Introspector.getBeanInfo(root.getClass());
        } catch (IntrospectionException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        }

        try {
            for( PropertyDescriptor pd : bi.getPropertyDescriptors() ){
                if( !pd.getName().equals(property) )continue;

                if( pd.getPropertyType()==null ){
                    return "may be index property";
                }

                Method m = pd.getWriteMethod();
                if( m==null )return "read only property";

                Class t = pd.getPropertyType();
                if( value==null ){
                    m.invoke(root, value);
                    if( assigned!=null ){
                        assigned.apply(value);
                    }
                }else{
                    if( t.isAssignableFrom(value.getClass()) ){
                        m.invoke(root, value);
                        if( assigned!=null ){
                            assigned.apply(value);
                        }
                    }else{
                        Object v = getTypeCastGraph().cast(value, t);
                        m.invoke(root, v);
                        if( assigned!=null ){
                            assigned.apply(v);
                        }
                    }
                }
                return null;
            }

            for( Field f : root.getClass().getFields() ){
                if( !f.getName().equals(property) )continue;

                Class t = f.getType();
                if( value==null ){
                    f.set(root, value);
                    if( assigned!=null ){
                        assigned.apply(value);
                    }
                }else{
                    if( t.isAssignableFrom(value.getClass()) ){
                        f.set(root, value);
                        if( assigned!=null ){
                            assigned.apply(value);
                        }
                    }else{
                        Object v = getTypeCastGraph().cast(value, t);
                        f.set(root, v);
                        if( assigned!=null ){
                            assigned.apply(v);
                        }
                    }
                }
                return null;
            }

            if( root instanceof Map ){
                ((Map)root).put(property, value);
                if( assigned!=null ){
                    assigned.apply(value);
                }
            }
        } catch (IllegalAccessException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        } catch (InvocationTargetException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return ex.getMessage();
        }

        return "property "+property+" not found";
    }
}
